const express = require('express');
const cors = require('cors');
const puppeteer = require('puppeteer');
const path = require('path');
const cluster = require('cluster');
const numCPUs = require('os').cpus().length;
const fs = require('fs').promises;
const fsSync = require('fs');
const sqlite3 = require('sqlite3').verbose();
const https = require('https');

// 🚀 性能优化：日志级别控制
const LOG_LEVEL = process.env.LOG_LEVEL || 'INFO'; // DEBUG, INFO, WARN, ERROR
const logLevels = { DEBUG: 0, INFO: 1, WARN: 2, ERROR: 3 };

// 简易文件日志：既输出到控制台，也落盘到 backend/logs/service.log
const FILE_LOG_PATH = path.join(__dirname, 'logs', 'service.log');
function ensureLogDir() {
    try {
        const dir = path.dirname(FILE_LOG_PATH);
        if (!fsSync.existsSync(dir)) {
            fsSync.mkdirSync(dir, { recursive: true });
        }
    } catch (e) {
        // 目录创建失败不影响控制台输出
    }
}
ensureLogDir();

function log(level, message, ...args) {
    if (logLevels[level] >= logLevels[LOG_LEVEL]) {
        const timestamp = new Date().toISOString();
        const line = `[${timestamp}] [${level}] ${message}`;
        // 控制台
        console.log(line, ...args);
        // 文件
        try {
            const formatted = args && args.length ? `${line} ${args.map(a => {
                try { return typeof a === 'string' ? a : JSON.stringify(a); } catch { return String(a); }
            }).join(' ')}` : line;
            fsSync.appendFileSync(FILE_LOG_PATH, formatted + "\n");
        } catch (e) {
            // 文件写入失败不影响主流程
        }
    }
}

// 进程级异常兜底：记录到文件，避免静默退出
process.on('uncaughtException', (err) => {
    try {
        const ts = new Date().toISOString();
        const stack = (err && err.stack) ? err.stack : String(err);
        fsSync.appendFileSync(FILE_LOG_PATH, `[${ts}] [FATAL] uncaughtException: ${stack}\n`);
    } catch {}
    console.error('uncaughtException:', err);
    // 不立即退出，尽量保持服务存活；如需强制重启请改用守护进程（pm2/nssm）
});

process.on('unhandledRejection', (reason, promise) => {
    try {
        const ts = new Date().toISOString();
        const msg = (reason && reason.stack) ? reason.stack : JSON.stringify(reason);
        fsSync.appendFileSync(FILE_LOG_PATH, `[${ts}] [ERROR] unhandledRejection: ${msg}\n`);
    } catch {}
    console.error('unhandledRejection:', reason);
});

// 🚀 性能优化：数据库连接池
class DatabasePool {
    constructor(dbPath, maxConnections = 5) {
        this.dbPath = dbPath;
        this.maxConnections = maxConnections;
        this.pool = [];
        this.activeConnections = 0;
        this.waitingQueue = [];
    }

    async getConnection() {
        return new Promise((resolve, reject) => {
            if (this.pool.length > 0) {
                const db = this.pool.pop();
                resolve(db);
                return;
            }

            if (this.activeConnections < this.maxConnections) {
                const db = new sqlite3.Database(this.dbPath, (err) => {
                    if (err) {
                        reject(err);
                        return;
                    }
                    this.activeConnections++;
                    resolve(db);
                });
                return;
            }

            this.waitingQueue.push({ resolve, reject });
        });
    }

    releaseConnection(db) {
        if (this.waitingQueue.length > 0) {
            const { resolve } = this.waitingQueue.shift();
            resolve(db);
        } else {
            this.pool.push(db);
        }
    }

    async closeAll() {
        const allConnections = [...this.pool];
        this.pool = [];
        
        for (const db of allConnections) {
            await new Promise(resolve => db.close(resolve));
        }
        this.activeConnections = 0;
    }
}

// 🚀 性能优化：配置缓存
class ProfileCache {
    constructor(ttl = 300000) { // 5分钟TTL
        this.cache = new Map();
        this.ttl = ttl;
    }

    set(key, value) {
        this.cache.set(key, {
            value,
            timestamp: Date.now()
        });
    }

    get(key) {
        const item = this.cache.get(key);
        if (!item) return null;
        
        if (Date.now() - item.timestamp > this.ttl) {
            this.cache.delete(key);
            return null;
        }
        
        return item.value;
    }

    clear() {
        this.cache.clear();
    }

    size() {
        return this.cache.size;
    }
}

// 初始化数据库连接池
const dbPath = path.join(__dirname, 'data', 'adsplus.db');
const credentialsDbPath = path.join(__dirname, 'profiles.db');
const mainDbPool = new DatabasePool(dbPath, 3);
const credentialsDbPool = new DatabasePool(credentialsDbPath, 2);

// 初始化配置缓存
const profileCache = new ProfileCache();

// 兼容性：保持原有的数据库连接变量（用于不需要优化的地方）
const db = new sqlite3.Database(dbPath);
const credentialsDb = new sqlite3.Database(credentialsDbPath);

// 🚀 性能优化：配置文件缓存
let profilesFileCache = null;
let profilesFileCacheTime = 0;
const PROFILES_CACHE_TTL = 60000; // 1分钟缓存

// 📁 读取配置文件函数（优化版）
async function loadProfilesFromFile() {
    try {
        // 检查缓存
        if (profilesFileCache && (Date.now() - profilesFileCacheTime) < PROFILES_CACHE_TTL) {
            log('DEBUG', `📁 使用缓存的配置文件数据: ${profilesFileCache.length} 个配置卡片`);
            return profilesFileCache;
        }

        const profilesPath = path.join(__dirname, 'data', 'profiles.json');
        log('DEBUG', `📁 读取配置文件: ${profilesPath}`);
        
        const data = await fs.readFile(profilesPath, 'utf8');
        const profiles = JSON.parse(data);
        
        // 更新缓存
        profilesFileCache = profiles;
        profilesFileCacheTime = Date.now();
        
        log('INFO', `✅ 成功读取 ${profiles.length} 个配置卡片`);
        return profiles;
    } catch (error) {
        log('WARN', `⚠️ 读取配置文件失败: ${error.message}`);
        return [];
    }
}

// 🔍 根据ID查找配置卡片（优化版）
async function findProfileById(profileId) {
    // 🚀 性能优化：检查缓存
    const cachedProfile = profileCache.get(profileId);
    if (cachedProfile) {
        log('DEBUG', `🚀 从缓存获取配置卡片: ${profileId}`);
        return cachedProfile;
    }

    log('DEBUG', `🔍 查找配置卡片: ${profileId}`);
    
    // 🚀 性能优化：使用数据库连接池
    const dbConnection = await mainDbPool.getConnection();
    
    try {
        // 首先尝试从主数据库的profiles表查找
        const dbProfile = await new Promise((resolve, reject) => {
            // 查询 adsplus.db 中的 profiles 表
            dbConnection.get(`SELECT id, name, account_name, account_email, account_password, account, 
                           start_url, user_agent, proxy, proxy_enabled, proxy_type, proxy_host, proxy_port, 
                           proxy_username, proxy_password, account_notes
                    FROM profiles WHERE id = ? OR name = ?`, [profileId, profileId], (err, row) => {
                if (err) {
                    log('ERROR', `❌ 查找配置卡片失败: ${err.message}`);
                    resolve(null);
                    return;
                }
                
                if (row) {
                    log('DEBUG', `✅ 从主数据库找到配置卡片 ${profileId}`);
                    
                    // 优先使用 account_email，然后是 account，最后是 account_name
                    const accountField = row.account_email || row.account || row.account_name;
                    log('DEBUG', `🔑 账号字段映射完成，使用: ${accountField || '未设置'}`);
                    
                    // 构建代理配置
                    let proxyConfig = null;
                    if (row.proxy_enabled && row.proxy_host) {
                        proxyConfig = {
                            type: row.proxy_type || 'http',
                            host: row.proxy_host,
                            port: row.proxy_port || 8080,
                            username: row.proxy_username,
                            password: row.proxy_password
                        };
                    } else if (row.proxy) {
                        try {
                            proxyConfig = JSON.parse(row.proxy);
                        } catch (e) {
                            log('WARN', `⚠️ 代理配置解析失败: ${e.message}`);
                        }
                    }
                    
                    const profile = {
                        id: profileId,
                        name: row.name,
                        account: accountField, // 使用正确的账号字段
                        accountName: row.account_name || accountField, // 添加 accountName 字段
                        accountEmail: row.account_email, // 保持兼容性
                        accountPassword: row.account_password,
                        startUrl: row.start_url || 'https://www.facebook.com/',
                        userAgent: row.user_agent || 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                        proxy: proxyConfig,
                        notes: row.account_notes || `配置卡片 ${row.name}`
                    };
                    
                    log('DEBUG', `✅ 配置卡片构建完成，账号: ${profile.account}`);
                    resolve(profile);
                } else {
                    log('DEBUG', `❌ 主数据库中未找到配置卡片: ${profileId}`);
                    resolve(null);
                }
            });
        });
    
        if (dbProfile) {
            // 🚀 性能优化：缓存结果
            profileCache.set(profileId, dbProfile);
            return dbProfile;
        }
        
        // 如果数据库中没有找到，尝试从文件中查找
        log('DEBUG', `📁 数据库中未找到配置卡片 ${profileId}，尝试从文件中查找...`);
        try {
            const profiles = await loadProfilesFromFile();
            const fileProfile = profiles.find(p => p.id === profileId);
            
            if (fileProfile) {
                log('DEBUG', `✅ 从文件找到配置卡片 ${profileId}`);
                log('DEBUG', `🔑 账号: ${fileProfile.account || '未设置'}`);
                
                // 🚀 性能优化：缓存结果
                profileCache.set(profileId, fileProfile);
                return fileProfile;
            } else {
                log('WARN', `⚠️ 文件中也未找到配置卡片: ${profileId}`);
                return null;
            }
        } catch (error) {
            log('ERROR', `❌ 从文件查找配置卡片失败: ${error.message}`);
            return null;
        }
    } finally {
        // 🚀 性能优化：释放数据库连接
        mainDbPool.releaseConnection(dbConnection);
    }
}



// 🔐 自动填充密码函数（优化版）
async function autoFillPasswordOnPage(page, profile, profileId) {
    try {
        log('DEBUG', `🔐 开始为页面自动填充密码 - Profile ID: ${profileId}`);
        
        // 直接使用 profile.accountName 作为账号
        const finalUsername = profile?.accountName || profile?.account || '';
        const accountPassword = profile?.accountPassword || '';
        
        if (!finalUsername || !accountPassword) {
            log('WARN', `⚠️ 配置卡片 ${profileId} 缺少账号或密码信息，跳过自动填充`);
            return false;
        }
        
        log('DEBUG', `🔐 准备自动填充凭证: ${finalUsername}`);
        
        // 等待页面加载完成
        log('DEBUG', '⏳ 等待页面加载完成...');
        await page.waitForTimeout(2000);
        log('DEBUG', '✅ 页面等待完成');
        
        // 获取当前页面URL用于调试
        const currentUrl = page.url();
        log('DEBUG', `📄 当前页面URL: ${currentUrl}`);
        
        // 在页面上下文中查找并填充登录表单
        const result = await page.evaluate(async (username, password) => {
            try {
                console.log(`🔑 使用配置的账号: ${username}`);
                
                // 🔍 增强的表单检测逻辑 - 查找用户名输入框
                const usernameSelectors = [
                    // 标准属性
                    'input[name="email"]',
                    'input[type="email"]',
                    'input[name="username"]',
                    'input[name="user"]',
                    'input[name="login"]',
                    'input[name="account"]',
                    'input[id="email"]',
                    'input[id="username"]',
                    'input[id="user"]',
                    'input[id="login"]',
                    'input[id="account"]',
                    // 自动完成属性
                    'input[autocomplete="username"]',
                    'input[autocomplete="email"]',
                    // 占位符文本
                    'input[placeholder*="email" i]',
                    'input[placeholder*="username" i]',
                    'input[placeholder*="用户名" i]',
                    'input[placeholder*="邮箱" i]',
                    'input[placeholder*="账号" i]',
                    // 类名
                    'input.username',
                    'input.email',
                    'input.login',
                    'input.account',
                    // 通用文本输入框（在登录表单中）
                    'form input[type="text"]:first-of-type',
                    'form input:not([type]):first-of-type'
                ];
                
                let usernameInput = null;
                for (const selector of usernameSelectors) {
                    try {
                        usernameInput = document.querySelector(selector);
                        if (usernameInput && usernameInput.type !== 'hidden') {
                            console.log(`🎯 找到用户名输入框: ${selector}`);
                            break;
                        }
                    } catch (e) {
                        // 忽略选择器错误，继续尝试下一个
                    }
                }
                
                // 🔍 增强的表单检测逻辑 - 查找密码输入框
                const passwordSelectors = [
                    // 标准密码字段
                    'input[type="password"]',
                    'input[name="password"]',
                    'input[name="pass"]',
                    'input[name="passwd"]',
                    'input[name="pwd"]',
                    'input[id="password"]',
                    'input[id="pass"]',
                    'input[id="passwd"]',
                    'input[id="pwd"]',
                    // 自动完成属性
                    'input[autocomplete="current-password"]',
                    'input[autocomplete="password"]',
                    // 占位符文本
                    'input[placeholder*="password" i]',
                    'input[placeholder*="密码" i]',
                    // 类名
                    'input.password',
                    'input.pass',
                    'input.pwd'
                ];
                
                let passwordInput = null;
                for (const selector of passwordSelectors) {
                    try {
                        passwordInput = document.querySelector(selector);
                        if (passwordInput) {
                            console.log(`🎯 找到密码输入框: ${selector}`);
                            break;
                        }
                    } catch (e) {
                        // 忽略选择器错误，继续尝试下一个
                    }
                }
                
                if (!usernameInput && !passwordInput) {
                    return { 
                        success: false, 
                        error: '未找到登录表单元素' 
                    };
                }
                
                let filledFields = [];
                

                
                // 填充用户名 - 模拟真实用户输入
                if (usernameInput && !usernameInput.value) {
                    usernameInput.focus();
                    
                    // 模拟逐字符输入
                    usernameInput.value = '';
                    for (let i = 0; i < username.length; i++) {
                        usernameInput.value += username[i];
                        usernameInput.dispatchEvent(new Event('input', { bubbles: true }));
                        await new Promise(resolve => setTimeout(resolve, 30));
                    }
                    
                    usernameInput.dispatchEvent(new Event('change', { bubbles: true }));
                    usernameInput.blur();
                    filledFields.push('username');
                    console.log(`✅ 已填充用户名: ${username}`);
                }
                
                // 等待一小段时间，模拟用户操作间隔
                await new Promise(resolve => setTimeout(resolve, 200));
                
                // 填充密码 - 模拟真实用户输入
                if (passwordInput && !passwordInput.value) {
                    passwordInput.focus();
                    
                    // 模拟逐字符输入密码
                    passwordInput.value = '';
                    for (let i = 0; i < password.length; i++) {
                        passwordInput.value += password[i];
                        passwordInput.dispatchEvent(new Event('input', { bubbles: true }));
                        await new Promise(resolve => setTimeout(resolve, 40));
                    }
                    
                    passwordInput.dispatchEvent(new Event('change', { bubbles: true }));
                    passwordInput.blur();
                    filledFields.push('password');
                    console.log(`✅ 已填充密码`);
                }
                

                
                return { 
                    success: true, 
                    filledFields: filledFields,
                    usernameFound: !!usernameInput,
                    passwordFound: !!passwordInput
                };
                
            } catch (error) {
                return { success: false, error: error.message };
            }
        }, finalUsername, accountPassword);
        
        if (result.success) {
            console.log(`✅ 自动填充完成: ${result.filledFields.join(', ')}`);
            console.log(`📊 表单元素: 用户名=${result.usernameFound}, 密码=${result.passwordFound}`);
            

            
            return true;
        } else {
            console.warn(`⚠️ 自动填充失败: ${result.error}`);
            return false;
        }
        
    } catch (error) {
        console.error(`❌ 自动填充密码失败:`, error.message);
        return false;
    }
}

const app = express();
const PORT = 9000;

// 🚀 性能优化配置
const BROWSER_POOL_SIZE = 10; // 浏览器实例池大小
const MAX_CONCURRENT_LAUNCHES = 3; // 最大并发启动数
const MEMORY_CLEANUP_INTERVAL = 60000; // 1分钟内存清理间隔

// 中间件
app.use(cors({
    origin: [
        'http://localhost:3000', 
        'http://localhost:8080', 
        'http://localhost:8081', 
        'http://127.0.0.1:8081', 
        'https://adplusgogogo.com',
        'http://adplusgogogo.com',
        'https://adsplusgo.top',
        'file://'
    ],
    credentials: true,
    methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
    allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With']
}));
app.use(express.json({ limit: '50mb' }));

// 📊 性能监控
let launchCount = 0;
let successCount = 0;
let errorCount = 0;
let totalLaunchTime = 0;

// 🗄️ 存储活跃的浏览器实例
const activeBrowsers = new Map();
const browserPool = [];
const launchQueue = [];
let currentLaunching = 0;

// 🚀 浏览器实例管理类（优化版）
// 提取邮箱前缀并安全化为文件名
function extractEmailPrefix(email) {
    if (!email || typeof email !== 'string') return null;
    const local = email.split('@')[0] || '';
    const sanitized = local.replace(/[^a-zA-Z0-9._-]/g, '_').toLowerCase();
    return sanitized || null;
}

class BrowserManager {
    constructor() {
        this.instances = new Map();
        this.pool = [];
        this.stats = {
            created: 0,
            destroyed: 0,
            active: 0,
            poolSize: 0
        };
    }

    // 🔍 检测并获取现有的用户数据目录
    findExistingUserDataDir(profileId, emailPrefix) {
        if (!profileId) return null;

        // 定义可能的目录路径（按优先级排序）
        const possiblePaths = [
            // 基于邮箱前缀的新格式（优先）
            ...(emailPrefix ? [
                path.join('D:\\adsbrowser\\browsers\\chrome', 'data', 'profiles', `${emailPrefix}_${profileId}`),
                path.join(__dirname, 'data', 'profiles', `${emailPrefix}_${profileId}`)
            ] : []),
            // 新的统一格式
            path.join('D:\\adsbrowser\\browsers\\chrome', 'data', 'profiles', `${profileId}_${profileId}`),
            // 当前使用的格式
            path.join('D:\\adsbrowser\\browsers\\chrome', 'data', 'profiles', String(profileId)),
            // data/profiles 下的格式
            path.join(__dirname, 'data', 'profiles', `${profileId}_${profileId}`),
            path.join(__dirname, 'data', 'profiles', String(profileId)),
            path.join(__dirname, 'data', 'profiles', `profile_${profileId}`)
        ];

        // 检查每个可能的路径
        for (const dirPath of possiblePaths) {
            if (fsSync.existsSync(dirPath)) {
                // 检查是否是有效的Chrome用户数据目录
                const defaultPath = path.join(dirPath, 'Default');
                if (fsSync.existsSync(defaultPath)) {
                    log('INFO', `🔍 发现现有用户数据目录: ${dirPath}`);
                    return dirPath;
                }
            }
        }

        return null;
    }

    // 🔧 获取或创建统一的用户数据目录
    getUnifiedUserDataDir(profileId, baseUserDataDir, emailPrefix) {
        if (!profileId) {
            return baseUserDataDir;
        }

        // 首先检查是否已存在目录
        const existingDir = this.findExistingUserDataDir(profileId, emailPrefix);
        if (existingDir) {
            log('INFO', `♻️ 复用现有用户数据目录: ${existingDir}`);
            return existingDir;
        }

        // 仅当调用方提供的目录位于 adsbrowser 路径下时才接受
        const adsbrowserProfilesRoot = path.join('D:\\adsbrowser\\browsers\\chrome', 'data', 'profiles');
        if (baseUserDataDir) {
            const normalizedBase = path.normalize(baseUserDataDir);
            const normalizedRoot = path.normalize(adsbrowserProfilesRoot);
            if (normalizedBase.startsWith(normalizedRoot)) {
                log('INFO', `🆕 使用调用方提供的用户数据目录(有效范围): ${baseUserDataDir}`);
                return baseUserDataDir;
            } else {
                log('WARN', `⚠️ 忽略外部用户数据目录: ${baseUserDataDir}，改用 adsbrowser 标准路径`);
            }
        }

        // 否则根据邮箱前缀或profileId构建统一格式目录
        const dirName = emailPrefix ? `${emailPrefix}_${profileId}` : `${profileId}_${profileId}`;
        const unifiedDir = path.join('D:\\adsbrowser\\browsers\\chrome', 'data', 'profiles', dirName);
        log('INFO', `🆕 创建新的用户数据目录: ${unifiedDir}`);
        return unifiedDir;
    }

    // 🧹 清理重复的用户数据目录
    async cleanupDuplicateDirectories(profileId, emailPrefix) {
        if (!profileId) return;

        const possiblePaths = [
            ...(emailPrefix ? [
                path.join(__dirname, '..', 'adsbrowser', 'browsers', 'chrome', 'data', 'profiles', `${emailPrefix}_${profileId}`),
                path.join(__dirname, 'data', 'profiles', `${emailPrefix}_${profileId}`)
            ] : []),
            path.join(__dirname, '..', 'adsbrowser', 'browsers', 'chrome', 'data', 'profiles', `${profileId}_${profileId}`),
            path.join(__dirname, '..', 'adsbrowser', 'browsers', 'chrome', 'data', 'profiles', String(profileId)),
            path.join(__dirname, 'data', 'profiles', `${profileId}_${profileId}`),
            path.join(__dirname, 'data', 'profiles', String(profileId)),
            path.join(__dirname, 'data', 'profiles', `profile_${profileId}`)
        ];

        const existingDirs = possiblePaths.filter(dirPath => fsSync.existsSync(dirPath));
        
        if (existingDirs.length > 1) {
            log('WARN', `🧹 发现 ${existingDirs.length} 个重复目录，profileId: ${profileId}`);
            
            // 选择最新的目录作为主目录（按修改时间）
            const dirStats = existingDirs.map(dir => ({
                path: dir,
                mtime: fsSync.statSync(dir).mtime
            }));
            
            dirStats.sort((a, b) => b.mtime - a.mtime);
            const primaryDir = dirStats[0].path;
            const duplicateDirs = dirStats.slice(1);
            
            log('INFO', `🎯 选择主目录: ${primaryDir}`);
            
            for (const duplicate of duplicateDirs) {
                log('INFO', `🗑️ 标记重复目录: ${duplicate.path}`);
                // 这里可以选择删除或重命名重复目录
                // 为了安全起见，我们只记录，不自动删除
            }
        }
    }

    // 🚀 性能优化：获取或创建浏览器实例
    async getBrowser(config) {
        const { executablePath, userDataDir, debugPort, profileId, proxy } = config;
        // 规范化用户数据目录，避免实例键包含外部路径
        const effectiveUserDataDir = this.getUnifiedUserDataDir(profileId, userDataDir, config.emailPrefix);
        
        // 🔧 优化实例键生成：区分代理和无代理模式
        let instanceKey;
        if (proxy && proxy.host && proxy.port) {
            // 有代理：使用代理信息作为键的一部分
            instanceKey = `${effectiveUserDataDir}_${profileId}_${proxy.host}:${proxy.port}`;
        } else {
            // 无代理：强制使用profileId确保独立实例
            instanceKey = `${effectiveUserDataDir}_${profileId}_no_proxy`;
        }
        
        log('DEBUG', `🔑 生成实例键: ${instanceKey} (代理: ${proxy ? `${proxy.host}:${proxy.port}` : '无'})`);

        // 检查是否已存在
        if (this.instances.has(instanceKey)) {
            const instance = this.instances.get(instanceKey);
            if (instance.browser && instance.browser.isConnected()) {
                log('DEBUG', `♻️ 复用浏览器实例: ${instanceKey}`);
                return instance;
            } else {
                // 清理无效实例
                this.instances.delete(instanceKey);
                this.stats.active--;
            }
        }

        // 创建新实例
        log('INFO', `🔨 创建新浏览器实例: ${instanceKey}`);
        const browser = await this.createBrowser(config);
        const instance = {
            browser,
            config,
            createdAt: Date.now(),
            lastUsed: Date.now(),
            profileId: config.profileId
        };

        this.instances.set(instanceKey, instance);
        this.stats.created++;
        this.stats.active++;

        return instance;
    }

    // 🚀 创建浏览器实例（优化版）
    async createBrowser(config) {
        const { executablePath, userDataDir, debugPort, proxy, userAgent, profileId, displayId } = config;
        
        // 🏷️ 确保windowTitle有默认值，避免undefined错误
        const windowTitle = config.windowTitle || `${displayId || profileId || 'Unknown'}`;
        
        // 🏷️ 调试日志：显示接收到的配置
        log('INFO', `🏷️ createBrowser接收到的配置: profileId=${profileId}, displayId=${displayId || 'N/A'}, windowTitle=${windowTitle}`);

        // 🔧 使用新的统一目录管理逻辑
        let actualUserDataDir = this.getUnifiedUserDataDir(profileId, userDataDir, config.emailPrefix);
        if (!actualUserDataDir || String(actualUserDataDir).trim().length === 0) {
            // 回退到统一规则下的默认目录（adsbrowser 路径）
            const dirName = config.emailPrefix ? `${config.emailPrefix}_${String(profileId || 'unknown')}` : `${String(profileId || 'unknown')}_${String(profileId || 'unknown')}`;
            actualUserDataDir = path.join(__dirname, '..', 'adsbrowser', 'browsers', 'chrome', 'data', 'profiles', dirName);
            log('WARN', `⚠️ userDataDir 为空，回退到默认目录: ${actualUserDataDir}`);
        }
        // 确保目录存在
        try {
            await fs.mkdir(actualUserDataDir, { recursive: true });
            log('DEBUG', `📁 用户数据目录已准备: ${actualUserDataDir}`);
        } catch (e) {
            log('ERROR', `❌ 创建用户数据目录失败: ${e.message}`);
        }
        log('DEBUG', `🔧 使用实际用户数据目录: ${actualUserDataDir} (profileId: ${profileId})`);

        // 🚀 性能优化：构建优化的Chrome启动参数
        let chromeArgs = [
            `--user-data-dir=${actualUserDataDir}`,
            `--remote-debugging-port=${debugPort}`,
            // 窗口配置 - 移除 --start-maximized 以允许 --window-size 参数正常工作
            // 安全相关
            '--disable-web-security',
            '--disable-popup-blocking',
            // 移除 --disable-sync 以允许历史记录保存
            '--disable-translate',
            // 性能优化
            '--disable-extensions',
            '--disable-background-timer-throttling',
            '--disable-component-update',
            '--disable-features=VizDisplayCompositor',
            '--disable-backgrounding-occluded-windows',
            '--disable-renderer-backgrounding',
            '--disable-field-trial-config',
            // 移除 --disable-back-forward-cache 以支持历史记录导航
            '--disable-background-networking',
            '--disable-client-side-phishing-detection',
            '--disable-features=TranslateUI',
            '--disable-ipc-flooding-protection',
            // 移除 --disable-prompt-on-repost 以启用密码保存提示
            '--disable-dev-shm-usage',
            // 系统相关
            '--no-sandbox',
            // 移除 --no-first-run 以允许正常的浏览器初始化
            '--no-default-browser-check',
            '--disable-default-apps',
            '--disable-hang-monitor',
            // 内存优化
            '--memory-pressure-off',
            '--max_old_space_size=4096',
            '--js-flags=--max-old-space-size=4096',
            // 🔐 启用原生密码管理器功能
            '--enable-password-manager-reauthentication',
            '--enable-save-password-bubble',
            '--enable-password-generation',
            '--enable-password-manager',
            '--password-store=basic',
            // 启用密码管理器相关功能
            '--enable-features=PasswordManager,PasswordGeneration,PasswordImport,PasswordExport',
            // 确保表单自动完成功能正常
            '--enable-autofill-keyboard-accessory-view',
            // 允许密码保存提示
            '--enable-password-save-bubble',
            // 确保密码管理器数据库正确初始化
            '--enable-password-manager-ui'
        ];

        // 添加代理配置
        if (proxy && proxy.host && proxy.port) {
            const proxyServer = `${proxy.host}:${proxy.port}`;
            chromeArgs.push(`--proxy-server=${proxyServer}`);
            log('DEBUG', `🔐 代理配置: ${proxyServer}`);
        }

        // 添加User Agent
        if (userAgent) {
            chromeArgs.push(`--user-agent=${userAgent}`);
            log('DEBUG', `🔧 User Agent: ${userAgent.substring(0, 50)}...`);
        }

        // 🖥️ 窗口大小配置 - 仅在明确指定时才设置固定大小
        if (config.windowSize && config.forceWindowSize) {
            // 将 "1920x1080" 格式转换为 "1920,1080" 格式
            const windowSize = config.windowSize.replace('x', ',');
            chromeArgs.push(`--window-size=${windowSize}`);
            log('DEBUG', `🖥️ 强制窗口大小: ${config.windowSize} -> ${windowSize}`);
        } else {
            // 让浏览器使用自然窗口大小，支持用户自由调整
            log('DEBUG', '🖥️ 使用自适应窗口大小，允许用户调整');
        }

        // 🏷️ 添加窗口标题配置 - 优化版，确保任务栏显示卡片ID且不出现灰色浏览器
        if (windowTitle) {
            // 基础窗口标题设置
            chromeArgs.push(`--window-name=${windowTitle}`);
            chromeArgs.push(`--app-name=${windowTitle}`);
            // 移除 --force-app-mode 和 --app 参数，这些会导致灰色浏览器
            
            // 强化任务栏显示的额外参数
            chromeArgs.push(`--class=${windowTitle}`);
            chromeArgs.push(`--name=${windowTitle}`);
            chromeArgs.push(`--title=${windowTitle}`);
            
            // 防止标题被覆盖的参数
            chromeArgs.push(`--disable-features=TranslateUI`);
            chromeArgs.push(`--disable-ipc-flooding-protection`);
            
            log('INFO', `🏷️ 优化窗口标题: ${windowTitle}`);
            log('INFO', `🏷️ 任务栏显示参数已添加（已修复灰色浏览器问题）`);
        }

        // 优先使用环境变量指定的 Chrome 路径（CHROME_PATH / PUPPETEER_EXECUTABLE_PATH），否则使用系统默认 Chrome
        const envChromePath = process.env.CHROME_PATH || process.env.PUPPETEER_EXECUTABLE_PATH;
        // 记录环境变量取值，便于定位为何仍然使用 115
        log('INFO', `🔧 环境变量 CHROME_PATH=${process.env.CHROME_PATH || '未设置'}`);
        log('INFO', `🔧 环境变量 PUPPETEER_EXECUTABLE_PATH=${process.env.PUPPETEER_EXECUTABLE_PATH || '未设置'}`);
        const effectiveExecutablePath = envChromePath || 'C:\\Program Files\\Google\\Chrome\\Application\\chrome.exe';
        log('INFO', `🔧 使用 Chrome 可执行文件: ${effectiveExecutablePath}`);
        const browser = await puppeteer.launch({
            executablePath: effectiveExecutablePath,
            headless: false,
            args: chromeArgs,
            userDataDir: actualUserDataDir,
            defaultViewport: null,
            ignoreDefaultArgs: [
                '--disable-extensions',
                '--disable-component-extensions-with-background-pages',
                '--disable-background-networking',
                '--disable-sync'
            ],
            handleSIGINT: false,
            handleSIGTERM: false,
            handleSIGHUP: false,
            timeout: 30000, // 30秒启动超时
            // 提升 CDP 协议超时，避免 Network.enable 等调用在慢环境下超时
            protocolTimeout: 120000
        });

        // 打印实际浏览器版本，确认是否仍为 115
        try {
            const ver = await browser.version();
            log('INFO', `🎯 实际 Chrome 版本: ${ver}`);
        } catch (verErr) {
            log('WARN', `⚠️ 读取 Chrome 版本失败: ${verErr.message}`);
        }

        // 全局提升协议超时，确保后续 CDP 调用更稳健
        try {
            if (typeof browser.setDefaultProtocolTimeout === 'function') {
                browser.setDefaultProtocolTimeout(120000);
                log('DEBUG', '⏱️ 已设置默认协议超时为 120000ms');
            }
        } catch (protoErr) {
            log('WARN', `⚠️ 设置默认协议超时失败: ${protoErr.message}`);
        }

        // 设置全局代理认证
        if (proxy && proxy.username && proxy.password) {
            browser.on('targetcreated', async (target) => {
                const page = await target.page();
                if (page) {
                    await page.authenticate({
                        username: proxy.username,
                        password: proxy.password
                    });
                }
            });
        }

        // 🔐 初始化密码管理器
        try {
            const pages = await browser.pages();
            if (pages.length > 0) {
                const page = pages[0];
                // 提升页面默认超时，减少慢网络/页面导致的超时
                try {
                    if (typeof page.setDefaultTimeout === 'function') {
                        page.setDefaultTimeout(60000);
                    }
                    if (typeof page.setDefaultNavigationTimeout === 'function') {
                        page.setDefaultNavigationTimeout(60000);
                    }
                    log('DEBUG', '⏱️ 已设置页面默认超时与导航超时为 60000ms');
                } catch (pageTimeoutErr) {
                    log('WARN', `⚠️ 设置页面默认超时失败: ${pageTimeoutErr.message}`);
                }
                
                // 启用密码管理器相关的 CDP 域
                const client = await page.target().createCDPSession();
                await client.send('Runtime.enable');
                await client.send('Page.enable');
                
                // 🚀 性能优化：按需初始化密码管理器状态
                await page.evaluate(() => {
                    // 设置密码保存相关的标志
                    window.__passwordManagerEnabled = true;
                    // 减少不必要的日志输出
                    if (window.navigator && window.navigator.credentials) {
                        window.__passwordManagerAvailable = true;
                    }
                });
                
                log('DEBUG', '✅ 密码管理器初始化完成');
            }
        } catch (initError) {
            log('WARN', '⚠️ 密码管理器初始化失败:', initError.message);
        }

        // 🏷️ 启动标题持续监控，确保任务栏始终显示卡片ID
        if (windowTitle) {
            try {
                await this.startTitleMonitoring(browser, windowTitle);
                log('INFO', `🏷️ 标题监控已启动: ${windowTitle}`);
            } catch (titleError) {
                log('WARN', '⚠️ 标题监控启动失败:', titleError.message);
            }
        }

        // 已移除：卡片信息标签页创建逻辑

        return browser;
    }

    // 🚀 性能优化：清理空闲实例
    async cleanupIdleInstances() {
        const now = Date.now();
        const toRemove = [];

        for (const [key, instance] of this.instances) {
            if (now - instance.lastUsed > BROWSER_IDLE_TIMEOUT) {
                toRemove.push(key);
            }
        }

        for (const key of toRemove) {
            await this.destroyInstance(key);
        }

        if (toRemove.length > 0) {
            log('INFO', `🧹 清理了 ${toRemove.length} 个空闲浏览器实例`);
        }
    }

    // 🚀 性能优化：销毁实例
    async destroyInstance(key) {
        const instance = this.instances.get(key);
        if (instance) {
            try {
                if (instance.browser && instance.browser.isConnected()) {
                    await instance.browser.close();
                }
                log('DEBUG', `🗑️ 浏览器实例已销毁: ${key}`);
            } catch (error) {
                log('WARN', `⚠️ 关闭浏览器实例失败: ${error.message}`);
            }

            this.instances.delete(key);
            this.stats.destroyed++;
            this.stats.active--;
        }
    }

    // 🏷️ 启动标题持续监控，确保任务栏始终显示卡片ID
    async startTitleMonitoring(browser, windowTitle) {
        try {
            const pages = await browser.pages();
            if (pages.length === 0) {
                // 已移除：为标题监控强制创建空白页；改为等待新页面
                log('DEBUG', 'ℹ️ 当前无页面，等待新页面创建以设置标题监控');
            } else {
                // 在现有页面上设置监控
                await this.setupTitleMonitoringOnPage(pages[0], windowTitle);
            }

            // 监听新页面创建事件
            browser.on('targetcreated', async (target) => {
                try {
                    if (target.type() === 'page') {
                        const page = await target.page();
                        if (page) {
                            await this.setupTitleMonitoringOnPage(page, windowTitle);
                        }
                    }
                } catch (error) {
                    log('WARN', '⚠️ 新页面标题监控设置失败:', error.message);
                }
            });

            log('INFO', `🏷️ 标题监控设置完成: ${windowTitle}`);
        } catch (error) {
            log('ERROR', '❌ 标题监控启动失败:', error.message);
            throw error;
        }
    }

    // 🏷️ 在页面上设置标题监控
    async setupTitleMonitoringOnPage(page, windowTitle) {
        try {
            // 注入标题监控脚本
            await page.evaluateOnNewDocument((title) => {
                // 立即设置标题
                document.title = title;
                
                // 创建持续监控
                let titleMonitorInterval;
                
                const startTitleMonitoring = () => {
                    // 清除之前的监控
                    if (titleMonitorInterval) {
                        clearInterval(titleMonitorInterval);
                    }
                    
                    // 普通页面：正常监控
                    titleMonitorInterval = setInterval(() => {
                        if (document.title !== title) {
                            document.title = title;
                            console.log(`🏷️ 标题已重置为: ${title}`);
                        }
                    }, 2000);
                };
                
                // 页面加载完成后启动监控
                if (document.readyState === 'loading') {
                    document.addEventListener('DOMContentLoaded', startTitleMonitoring);
                } else {
                    startTitleMonitoring();
                }
                
                // 监听标题变化
                const observer = new MutationObserver((mutations) => {
                    mutations.forEach((mutation) => {
                        if (mutation.type === 'childList' && mutation.target.tagName === 'TITLE') {
                            if (document.title !== title) {
                                document.title = title;
                                console.log(`🏷️ 标题被重置: ${title}`);
                            }
                        }
                    });
                });
                
                // 开始观察title元素变化
                const titleElement = document.querySelector('title');
                if (titleElement) {
                    observer.observe(titleElement, { childList: true, characterData: true });
                }
                
                // 监听页面标题变化事件
                document.addEventListener('DOMContentLoaded', () => {
                    const titleElement = document.querySelector('title');
                    if (titleElement) {
                        observer.observe(titleElement, { childList: true, characterData: true });
                    }
                });
                
                console.log(`🏷️ 标题监控脚本已注入: ${title}`);
            }, windowTitle);

            log('DEBUG', `🏷️ 页面标题监控已设置: ${windowTitle}`);
        } catch (error) {
            log('WARN', '⚠️ 页面标题监控设置失败:', error.message);
        }
    }

    // 已移除：关闭初始空白页和信息页逻辑

    // 已移除：卡片信息标签页创建函数

    // 已移除：卡片信息HTML生成函数

    // 获取统计信息
    getStats() {
        return {
            ...this.stats,
            poolSize: this.pool.length,
            instanceCount: this.instances.size
        };
    }

    // 清理所有实例
    async cleanup() {
        const keys = Array.from(this.instances.keys());
        for (const key of keys) {
            await this.destroyInstance(key);
        }
    }
}

// 创建浏览器管理器实例
const browserManager = new BrowserManager();

// 🚀 启动队列管理
class LaunchQueue {
    constructor(maxConcurrent = MAX_CONCURRENT_LAUNCHES) {
        this.queue = [];
        this.running = 0;
        this.maxConcurrent = maxConcurrent;
    }

    async add(task) {
        return new Promise((resolve, reject) => {
            this.queue.push({ task, resolve, reject });
            this.process();
        });
    }

    async process() {
        if (this.running >= this.maxConcurrent || this.queue.length === 0) {
            return;
        }

        this.running++;
        const { task, resolve, reject } = this.queue.shift();

        try {
            const result = await task();
            resolve(result);
        } catch (error) {
            reject(error);
        } finally {
            this.running--;
            this.process(); // 处理下一个任务
        }
    }

    getStats() {
        return {
            queueLength: this.queue.length,
            running: this.running,
            maxConcurrent: this.maxConcurrent
        };
    }
}

const launchQueueManager = new LaunchQueue();

// ❤️ 健康检查端点
app.get('/api/health', (req, res) => {
    const memUsage = process.memoryUsage();
    
    res.json({ 
        status: 'ok', 
        timestamp: new Date().toISOString(),
        service: 'AdsPlus Go Puppeteer Service',
        version: '1.1.0',
        performance: {
            activeBrowsers: activeBrowsers.size,
            browserInstances: browserManager.getStats(),
            launchQueue: launchQueueManager.getStats(),
            memory: {
                used: Math.round(memUsage.heapUsed / 1024 / 1024),
                total: Math.round(memUsage.heapTotal / 1024 / 1024),
                external: Math.round(memUsage.external / 1024 / 1024)
            },
            stats: {
                launches: launchCount,
                successes: successCount,
                errors: errorCount,
                avgLaunchTime: launchCount > 0 ? Math.round(totalLaunchTime / launchCount) : 0
            }
        }
    });
});

// ❤️ 根路径健康检查端点（用于前端状态检查）
app.get('/health', (req, res) => {
    res.json({ 
        status: 'ok', 
        timestamp: new Date().toISOString(),
        service: 'AdsPlus Go Backend Service',
        version: '1.1.0'
    });
});

// 🧹 清理重复目录端点
app.post('/api/cleanup-duplicates', async (req, res) => {
    try {
        const { profileId } = req.body;
        
        if (!profileId) {
            return res.status(400).json({ 
                success: false, 
                error: '缺少 profileId 参数' 
            });
        }

        await browserManager.cleanupDuplicateDirectories(profileId);
        
        res.json({ 
            success: true, 
            message: `已检查并清理 profileId: ${profileId} 的重复目录` 
        });
    } catch (error) {
        console.error('清理重复目录失败:', error);
        res.status(500).json({ 
            success: false, 
            error: error.message 
        });
    }
});

// 📊 性能统计端点
app.get('/api/stats', (req, res) => {
    res.json({
        success: true,
        data: {
            browserManager: browserManager.getStats(),
            launchQueue: launchQueueManager.getStats(),
            performance: {
                launches: launchCount,
                successes: successCount,
                errors: errorCount,
                avgLaunchTime: launchCount > 0 ? Math.round(totalLaunchTime / launchCount) : 0
            },
            memory: process.memoryUsage(),
            uptime: process.uptime()
        }
    });
});

// 🚀 启动浏览器端点 (优化版)
app.post('/api/launch-browser', async (req, res) => {
    const startTime = Date.now();
    launchCount++;
    
    const { profileId, profile, proxy, startUrls, cookies, userAgent, executablePath, chrome115Config } = req.body;
    
    // 🏷️ 提取窗口标题配置
    const windowTitle = chrome115Config?.windowTitle;
    console.log(`🏷️ API端点提取到的windowTitle: ${windowTitle}`);
    
    try {
        console.log(`🚀 启动浏览器配置 ${profileId}...`);

        // 🔍 从文件中读取完整的配置信息
        const actualProfile = await findProfileById(profileId);
        if (!actualProfile) {
            throw new Error(`配置卡片 ${profileId} 不存在`);
        }

        // 使用从文件读取的配置信息，而不是前端传递的profile
        const finalProfile = actualProfile;
        const finalProxy = proxy || actualProfile.proxy;
        const finalStartUrls = startUrls || [actualProfile.startUrl].filter(Boolean);
        const finalUserAgent = userAgent || actualProfile.userAgent;

        console.log(`📋 使用配置信息:`);
        console.log(`📧 账号邮箱: ${finalProfile.accountEmail || '未设置'}`);
        console.log(`🔑 密码状态: ${finalProfile.accountPassword ? '已设置' : '未设置'}`);
        console.log(`🌐 启动网站: ${finalStartUrls.join(', ') || '未设置'}`);

        // 使用启动队列管理并发
        const result = await launchQueueManager.add(async () => {
            // 🔧 设置默认参数，避免undefined（使用邮箱前缀+profileId作为目录名）
            const emailPrefix = extractEmailPrefix(finalProfile.accountEmail || finalProfile.account_email);
            const dirName = emailPrefix ? `${emailPrefix}_${String(profileId)}` : String(profileId);
            const defaultUserDataDir = path.join('D:\\adsbrowser\\browsers\\chrome', 'data', 'profiles', dirName);
            // 基于profileId生成唯一端口（处理字符串profileId）
            const profileHash = profileId ? String(profileId).split('').reduce((a, b) => {
                a = ((a << 5) - a) + b.charCodeAt(0);
                return a & a;
            }, 0) : 0;
            const defaultDebugPort = 9222 + Math.abs(profileHash) % 1000;
            const defaultExecutablePath = (process.env.CHROME_PATH || process.env.PUPPETEER_EXECUTABLE_PATH || 'C:\\Program Files\\Google\\Chrome\\Application\\chrome.exe');
            
            // 配置浏览器参数
            const config = {
                profileId,
                executablePath: executablePath || defaultExecutablePath,
                userDataDir: req.body.userDataDir || defaultUserDataDir,
                debugPort: req.body.debugPort || defaultDebugPort,
                proxy: finalProxy,
                userAgent: finalUserAgent,
                windowTitle: windowTitle, // 🏷️ 添加窗口标题配置
                windowSize: finalProfile.resolution || '1200x800', // 🖥️ 添加窗口大小配置
                emailPrefix
            };

            console.log(`🔧 浏览器配置: executablePath=${config.executablePath}, userDataDir=${config.userDataDir}, debugPort=${config.debugPort}`);

            // 🔧 使用统一的目录管理逻辑
            const actualUserDataDir = browserManager.getUnifiedUserDataDir(profileId, config.userDataDir, emailPrefix);
            
            // 🧹 检查并清理重复目录
            await browserManager.cleanupDuplicateDirectories(profileId, config.emailPrefix);
            
            // 🍪 解析Cookie数据并预注入到数据库
            let parsedCookies = null;
            if (cookies && (typeof cookies === 'string' ? cookies.trim().length > 0 : cookies.length > 0)) {
                console.log(`🍪 检测到Cookie数据，准备预注入到Chrome数据库...`);
                console.log(`🍪 Cookie数据类型: ${typeof cookies}, 内容预览: ${typeof cookies === 'string' ? cookies.substring(0, 100) : `${cookies.length} 个cookie对象`}`);
                
                try {
                    // 解析Cookie数据
                    parsedCookies = cookies;
                    if (typeof cookies === 'string') {
                        parsedCookies = parseCookieString(cookies, finalStartUrls);
                        console.log(`🔧 Cookie字符串解析结果: ${parsedCookies.length} 个cookie`);
                    }
                    
                    if (parsedCookies && parsedCookies.length > 0) {
                        log('INFO', `🍪 开始预注入 ${parsedCookies.length} 个Cookie到Chrome数据库...`);
                        
                        // 确保用户数据目录存在
                        await fs.mkdir(actualUserDataDir, { recursive: true });
                        log('DEBUG', `📁 用户数据目录已创建: ${actualUserDataDir}`);
                        
                        // 🚀 性能优化：预注入Cookie到Chrome数据库
                        await preinjectCookiesToDatabase(actualUserDataDir, parsedCookies);
                        log('INFO', `✅ Cookie预注入完成，浏览器启动后将自动读取这些Cookie`);
                    } else {
                        log('DEBUG', `ℹ️ 解析后没有有效的Cookie数据`);
                        parsedCookies = null;
                    }
                } catch (cookieError) {
                    log('WARN', `⚠️ Cookie预注入失败: ${cookieError.message}`);
                    log('WARN', `⚠️ 将回退到页面级别的动态注入`);
                    // 即使预注入失败，也继续启动浏览器，使用动态注入作为备用
                    if (typeof cookies === 'string') {
                        try {
                            parsedCookies = parseCookieString(cookies, startUrls);
                        } catch (parseError) {
                            log('WARN', `⚠️ Cookie解析也失败: ${parseError.message}`);
                            parsedCookies = null;
                        }
                    }
                }
            } else {
                log('DEBUG', `ℹ️ 没有Cookie数据需要注入`);
            }
            
            // 获取或创建浏览器实例（在Cookie预注入之后）
            const browserInstance = await browserManager.getBrowser(config);
            browserInstance.lastUsed = Date.now();

            // 已移除：启动后清理初始页面逻辑
            

            
            // 注意：Cookie验证将在网站导航后进行，因为Cookie需要在正确的域名下才能被读取

            // 存储活跃浏览器
            activeBrowsers.set(profileId, {
                browser: browserInstance.browser,
                profileId,
                startTime: Date.now(),
                debugPort: config.debugPort,
                proxy: finalProxy,
                userAgent: config.userAgent
            });

            // 处理启动网站
            if (finalStartUrls && finalStartUrls.length > 0) {
                console.log(`🌐 准备打开启动网站: ${finalStartUrls.join(', ')}`);
                
                // 创建新标签页打开网站
                for (const url of finalStartUrls.slice(0, 3)) { // 限制最多3个网站
                    try {
                        const page = await browserInstance.browser.newPage();
                        
                        // 设置页面级别的配置
                        if (config.userAgent) {
                            await page.setUserAgent(config.userAgent);
                        }
                        
                        // 🚀 性能优化：简化Cookie动态注入
                        if (parsedCookies && parsedCookies.length > 0) {
                            log('DEBUG', `🍪 开始通过CDP协议动态注入 ${parsedCookies.length} 个Cookie...`);
                            
                            let injectedCount = 0;
                            
                            for (const cookie of parsedCookies) {
                                try {
                                    // 构建符合Puppeteer要求的Cookie对象
                                    const puppeteerCookie = {
                                        name: cookie.name,
                                        value: cookie.value,
                                        domain: cookie.domain || new URL(url).hostname,
                                        path: cookie.path || '/',
                                        secure: cookie.secure !== undefined ? cookie.secure : url.startsWith('https://'),
                                        httpOnly: cookie.httpOnly || false,
                                        sameSite: cookie.sameSite || 'None'
                                    };
                                    
                                    // 设置过期时间
                                    if (cookie.expirationDate) {
                                        puppeteerCookie.expires = cookie.expirationDate;
                                    } else if (cookie.expiry) {
                                        puppeteerCookie.expires = cookie.expiry;
                                    }
                                    
                                    // 注入Cookie
                                    await page.setCookie(puppeteerCookie);
                                    injectedCount++;
                                    
                                } catch (cookieError) {
                                    log('WARN', `❌ Cookie注入失败 ${cookie.name}: ${cookieError.message}`);
                                }
                            }
                            
                            log('INFO', `🍪 Cookie动态注入完成: ${injectedCount}/${parsedCookies.length} 成功`);
                        }
                        
                        // 导航到网站
                        try {
                            await page.goto(url, { 
                                waitUntil: 'networkidle2',
                                timeout: 30000 
                            });
                            console.log(`✅ 已打开网站: ${url}`);
                            
                            // 🔐 自动填充密码（如果是登录页面）
                            try {
                                console.log(`🔐 尝试自动填充登录表单...`);
                                await autoFillPasswordOnPage(page, finalProfile, profileId);
                            } catch (autoFillError) {
                                console.warn(`⚠️ 自动填充失败: ${autoFillError.message}`);
                            }
                            
                            // 🔐 原生密码管理器已通过CDP实现，无需额外脚本注入
                            console.log('✅ 使用浏览器原生密码管理器，自动填充已执行');
                            
                        } catch (navigationError) {
                            console.warn(`⚠️ 打开网站失败 ${url}: ${navigationError.message}`);
                            // Cookie注入成功，网站访问失败不影响浏览器启动
                            continue;
                        }
                        
                        // 🍪 验证Cookie是否成功注入（在目标域名页面验证）
                        if (parsedCookies && parsedCookies.length > 0) {
                            try {
                                const pageCookies = await page.cookies();
                                console.log(`🍪 页面加载后检测到 ${pageCookies.length} 个Cookie`);
                                
                                // 检查预期的Cookie是否存在
                                const expectedCookieNames = ['datr', 'fr', 'sb', 'c_user', 'xs'];
                                let verifiedCount = 0;
                                
                                for (const expectedName of expectedCookieNames) {
                                    const foundCookie = pageCookies.find(c => c.name === expectedName);
                                    if (foundCookie) {
                                        console.log(`✅ Cookie验证成功: ${expectedName}`);
                                        verifiedCount++;
                                    } else {
                                        console.log(`⚠️ Cookie验证失败: ${expectedName} 未找到`);
                                    }
                                }
                                
                                // 检查所有注入的Cookie
                                for (const expectedCookie of parsedCookies.slice(0, 10)) { // 检查前10个
                                    const foundCookie = pageCookies.find(c => 
                                        c.name === expectedCookie.name && 
                                        c.domain === expectedCookie.domain
                                    );
                                    if (!foundCookie && !expectedCookieNames.includes(expectedCookie.name)) {
                                        console.log(`⚠️ 注入Cookie未找到: ${expectedCookie.name} (domain: ${expectedCookie.domain})`);
                                    }
                                }
                                
                                console.log(`🍪 关键Cookie验证完成: ${verifiedCount}/${expectedCookieNames.length} 个关键Cookie验证成功`);
                                
                                if (verifiedCount === 0) {
                                    console.warn(`⚠️ 所有关键Cookie验证都失败，可能需要检查Cookie格式或域名设置`);
                                } else if (verifiedCount < expectedCookieNames.length) {
                                    console.warn(`⚠️ 部分关键Cookie验证失败，可能影响功能正常使用`);
                                } else {
                                    console.log(`✅ 所有关键Cookie验证成功！`);
                                }
                                
                            } catch (verifyError) {
                                console.warn(`⚠️ Cookie验证失败:`, verifyError.message);
                            }
                        }
                    } catch (pageError) {
                        console.warn(`⚠️ 打开网站失败 ${url}:`, pageError.message);
                    }
                }
            }

            return {
                profileId,
                debugPort: config.debugPort,
                startTime: browserInstance.createdAt,
                userAgent: config.userAgent,
                proxyConfig: proxy
            };
        });

        const launchTime = Date.now() - startTime;
        totalLaunchTime += launchTime;
        successCount++;

        console.log(`✅ 浏览器启动成功: ${profileId} (${launchTime}ms)`);

        res.json({
            success: true,
            message: '浏览器启动成功',
            data: result,
            launchTime,
            timestamp: new Date().toISOString()
        });

    } catch (error) {
        const launchTime = Date.now() - startTime;
        errorCount++;
        
        console.error(`❌ 启动浏览器失败: ${profileId}`, error);
        
        res.status(500).json({
            success: false,
            message: '浏览器启动失败',
            error: error.message,
            profileId,
            launchTime,
            timestamp: new Date().toISOString()
        });
    }
});

// 🛑 停止浏览器端点
app.post('/api/stop-browser', async (req, res) => {
    const { profileId } = req.body;
    
    try {
        const browserData = activeBrowsers.get(profileId);
        if (!browserData) {
            return res.status(404).json({
                success: false,
                message: '未找到运行中的浏览器实例'
            });
        }

        // 关闭浏览器
        if (browserData.browser && browserData.browser.isConnected()) {
            await browserData.browser.close();
        }

        // 从活跃列表中移除
        activeBrowsers.delete(profileId);

        console.log(`🛑 浏览器已停止: ${profileId}`);

        res.json({
            success: true,
            message: '浏览器停止成功',
            profileId,
            timestamp: new Date().toISOString()
        });

    } catch (error) {
        console.error(`❌ 停止浏览器失败: ${profileId}`, error);
        
        res.status(500).json({
            success: false,
            message: '停止浏览器失败',
            error: error.message,
            profileId,
            timestamp: new Date().toISOString()
        });
    }
});

// 🔄 重启浏览器端点
app.post('/api/restart-browser', async (req, res) => {
    const { profileId } = req.body;
    
    try {
        // 先停止
        const stopResult = await new Promise((resolve) => {
            const mockReq = { body: { profileId } };
            const mockRes = {
                status: () => mockRes,
                json: resolve
            };
            // 调用停止逻辑 (简化版)
        });

        // 等待一秒
        await new Promise(resolve => setTimeout(resolve, 1000));

        // 重新启动 (需要完整的配置信息)
        res.json({
            success: true,
            message: '浏览器重启成功，请重新发送启动请求',
            profileId,
            timestamp: new Date().toISOString()
        });

    } catch (error) {
        console.error(`❌ 重启浏览器失败: ${profileId}`, error);
        
        res.status(500).json({
            success: false,
            message: '重启浏览器失败',
            error: error.message,
            profileId,
            timestamp: new Date().toISOString()
        });
    }
});

// 📋 获取活跃浏览器列表
app.get('/api/browsers', (req, res) => {
    const browsers = Array.from(activeBrowsers.entries()).map(([profileId, data]) => ({
        profileId,
        debugPort: data.debugPort,
        startTime: data.startTime,
        uptime: Date.now() - data.startTime,
        proxy: data.proxy ? `${data.proxy.host}:${data.proxy.port}` : null,
        userAgent: data.userAgent ? data.userAgent.substring(0, 50) + '...' : null
    }));

    res.json({
        success: true,
        data: browsers,
        count: browsers.length,
        timestamp: new Date().toISOString()
    });
});

// 🔑 获取登录凭据端点
app.get('/api/credentials/:domain', (req, res) => {
    const { domain } = req.params;
    const { profileId } = req.query;
    
    try {
        console.log(`🔍 查询登录凭据: ${domain} (Profile: ${profileId})`);
        
        const db = new sqlite3.Database('./profiles.db');
        
        // 查询该域名的登录凭据
        const sql = `
            SELECT id, username, password_encrypted, website_name, form_selectors, is_default, auto_login
            FROM login_credentials 
            WHERE website_domain LIKE ? AND profile_id = ?
            ORDER BY is_default DESC, id ASC
        `;
        
        db.all(sql, [`%${domain}%`, profileId || 'test-profile-001'], (err, rows) => {
            if (err) {
                console.error('❌ 查询登录凭据失败:', err.message);
                return res.status(500).json({
                    success: false,
                    message: '查询登录凭据失败',
                    error: err.message
                });
            }
            
            // 解密密码并格式化数据
            const credentials = rows.map(row => ({
                id: row.id,
                username: row.username,
                password: 'test-password', // 简化处理，实际应该解密
                website_name: row.website_name,
                form_selectors: row.form_selectors ? JSON.parse(row.form_selectors) : null,
                is_default: row.is_default === 1,
                auto_login: row.auto_login === 1
            }));
            
            console.log(`✅ 找到 ${credentials.length} 个登录凭据`);
            
            res.json({
                success: true,
                data: credentials,
                domain,
                profileId: profileId || 'test-profile-001',
                timestamp: new Date().toISOString()
            });
            
            db.close();
        });
        
    } catch (error) {
        console.error('❌ 获取登录凭据失败:', error);
        res.status(500).json({
            success: false,
            message: '获取登录凭据失败',
            error: error.message
        });
    }
});

// 🍪 验证浏览器Cookie端点
app.post('/api/validate-cookies', async (req, res) => {
    const { profileId } = req.body;
    
    try {
        console.log(`🔍 开始验证 Profile ${profileId} 的Cookie...`);
        
        // 从activeBrowsers获取浏览器实例
        const browserData = activeBrowsers.get(profileId);
        if (!browserData) {
            return res.status(404).json({
                success: false,
                profileId,
                status: 'not_found',
                message: '未找到运行中的浏览器实例',
                cookies: []
            });
        }
        
        const { browser } = browserData;
        
        // 检查浏览器连接状态
        if (!browser || !browser.isConnected()) {
            return res.json({
                success: false,
                profileId,
                status: 'disconnected',
                message: '浏览器实例已断开连接',
                cookies: []
            });
        }
        
        // 获取所有页面
        const pages = await browser.pages();
        
        if (pages.length === 0) {
            return res.json({
                success: true,
                profileId,
                status: 'no_pages',
                message: '浏览器中没有打开的页面',
                totalCookies: 0,
                cookies: [],
                analysis: {
                    total: 0,
                    domains: [],
            
                    importantCookies: [],
                    sessionCookies: 0,
                    persistentCookies: 0
                }
            });
        }
        
        // 使用第一个页面进行验证
        const page = pages[0];
        const url = page.url();
        
        // 获取所有Cookie
        const cookies = await page.cookies();
        
        // 分析Cookie
        const analysis = {
            total: cookies.length,
            domains: new Set(),
    
            importantCookies: [],
            sessionCookies: 0,
            persistentCookies: 0
        };
        
        // 重要Cookie名称
        const importantCookieNames = ['c_user', 'xs', 'datr', 'sb', 'fr', 'wd', 'presence'];
        
        cookies.forEach(cookie => {
            // 统计域名
            analysis.domains.add(cookie.domain);
            
            // 检查重要Cookie
            if (importantCookieNames.includes(cookie.name)) {
                analysis.importantCookies.push(cookie.name);
            }
            
            // 统计Cookie类型
            if (cookie.expires && cookie.expires > 0) {
                analysis.persistentCookies++;
            } else {
                analysis.sessionCookies++;
            }
        });
        
        analysis.domains = Array.from(analysis.domains);
        
        console.log(`✅ Profile ${profileId} Cookie验证完成: ${cookies.length}个Cookie`);
        
        res.json({
            success: true,
            profileId,
            status: 'success',
            url,
            totalCookies: cookies.length,
            cookies: cookies.map(cookie => ({
                name: cookie.name,
                value: cookie.value.substring(0, 20) + '...',
                domain: cookie.domain,
                path: cookie.path,
                secure: cookie.secure,
                httpOnly: cookie.httpOnly
            })),
            analysis,
            timestamp: new Date().toISOString()
        });
        
    } catch (error) {
        console.error(`❌ 验证 Profile ${profileId} Cookie失败:`, error.message);
        
        res.status(500).json({
            success: false,
            profileId,
            status: 'error',
            message: `验证失败: ${error.message}`,
            cookies: [],
            timestamp: new Date().toISOString()
        });
    }
});

// 🚀 启动Puppeteer浏览器端点 (兼容enhanced.html的API调用)
app.post('/api/browser/launch-puppeteer', async (req, res) => {
    const startTime = Date.now();
    launchCount++;
    
    const { profileId, profile, proxy, startUrls, cookies, userAgent, executablePath, userDataDir, url, startUrl, chrome115Config } = req.body;
    
    // 🏷️ 提取窗口标题配置
    const windowTitle = chrome115Config?.windowTitle;
    
    // 📁 从文件读取完整的配置信息
    let finalProfile = profile;
    let finalProxy = proxy;
    let finalUserAgent = userAgent;
    
    try {
        const fileProfile = await findProfileById(profileId);
        if (fileProfile) {
            console.log(`📁 找到配置卡片 ${profileId}，使用文件中的配置信息`);
            finalProfile = fileProfile;
            finalProxy = fileProfile.proxy || proxy;
            finalUserAgent = fileProfile.userAgent || userAgent;
            console.log(`🔑 账号: ${fileProfile.account || '未设置'}`);
            console.log(`🔑 密码状态: ${fileProfile.accountPassword ? '已设置' : '未设置'}`);
            console.log(`🌐 代理设置: ${finalProxy ? JSON.stringify(finalProxy) : '未设置'}`);
            console.log(`🔧 用户代理: ${finalUserAgent || '默认'}`);
        } else {
            console.log(`⚠️ 配置卡片 ${profileId} 在文件中未找到，使用传递的参数`);
        }
    } catch (error) {
        console.error(`❌ 读取配置卡片 ${profileId} 失败:`, error.message);
        console.log(`⚠️ 使用传递的参数作为备用`);
    }
    
    // 🔧 支持多种URL参数格式
    let finalStartUrls = startUrls;
    if (!finalStartUrls && (url || startUrl)) {
        finalStartUrls = [url || startUrl];
    }
    
    // 🔧 确保debugPort有默认值，避免undefined导致路径错误
    const debugPort = req.body.debugPort || 9222;
    console.log(`🔧 使用调试端口: ${debugPort} (${req.body.debugPort ? '用户指定' : '默认值'})`);
    
    try {
        console.log(`🚀 启动Puppeteer浏览器配置 ${profileId}...`);
        console.log(`🌐 接收到的启动网站:`, finalStartUrls);

        // 使用启动队列管理并发
        const result = await launchQueueManager.add(async () => {
            // 配置浏览器参数
            const config = {
                profileId,
                executablePath,
                userDataDir,
                debugPort,
                proxy: finalProxy,
                userAgent: finalUserAgent,
                windowTitle: windowTitle, // 🏷️ 添加窗口标题配置
                emailPrefix: extractEmailPrefix(finalProfile?.accountEmail || finalProfile?.account_email)
            };

            // 🔧 计算实际的用户数据目录（优先使用邮箱前缀+profileId的统一规则）
            const actualUserDataDir = browserManager.getUnifiedUserDataDir(profileId, userDataDir, config.emailPrefix);
            
            // 🍪 解析Cookie数据（不再预注入到数据库，改为动态注入）
            let parsedCookies = null;
            if (cookies && (typeof cookies === 'string' ? cookies.trim().length > 0 : cookies.length > 0)) {
                console.log(`🍪 检测到Cookie数据，准备动态注入...`);
                console.log(`🍪 Cookie数据类型: ${typeof cookies}, 内容预览: ${typeof cookies === 'string' ? cookies.substring(0, 100) : `${cookies.length} 个cookie对象`}`);
                
                try {
                    // 解析Cookie数据
                    parsedCookies = cookies;
                    if (typeof cookies === 'string') {
                        parsedCookies = parseCookieString(cookies, startUrls);
                        console.log(`🔧 Cookie字符串解析结果: ${parsedCookies.length} 个cookie`);
                    }
                    
                    if (parsedCookies && parsedCookies.length > 0) {
                        console.log(`✅ Cookie解析完成: ${parsedCookies.length} 个cookie，将在页面加载时动态注入`);
                    } else {
                        console.log(`ℹ️ 解析后没有有效的Cookie数据`);
                        parsedCookies = null;
                    }
                } catch (cookieError) {
                    console.warn(`⚠️ Cookie解析失败:`, cookieError.message);
                    parsedCookies = null;
                }
            } else {
                console.log(`ℹ️ 没有Cookie数据需要注入`);
            }

            // 获取或创建浏览器实例
            const browserInstance = await browserManager.getBrowser(config);
            browserInstance.lastUsed = Date.now();

            // 已移除：启动后清理初始页面逻辑



            // 存储活跃浏览器
            activeBrowsers.set(profileId, {
                browser: browserInstance.browser,
                profileId,
                startTime: Date.now(),
                debugPort,
                proxy: finalProxy,
                userAgent: config.userAgent
            });

            // 处理启动网站
            if (finalStartUrls && finalStartUrls.length > 0) {
                console.log(`🌐 准备打开启动网站: ${finalStartUrls.join(', ')}`);
                
                // 创建新标签页打开网站
                for (const url of finalStartUrls.slice(0, 3)) { // 限制最多3个网站
                    try {
                        const page = await browserInstance.browser.newPage();
                        
                        // 设置页面级别的配置
                        if (config.userAgent) {
                            await page.setUserAgent(config.userAgent);
                        }
                        
                        // 🍪 动态注入Cookie（在导航之前）
                        if (parsedCookies && parsedCookies.length > 0) {
                            console.log(`🍪 开始为页面 ${url} 动态注入 ${parsedCookies.length} 个Cookie...`);
                            
                            // 先导航到目标域名的空白页面以建立正确的上下文
                            try {
                                const targetUrl = new URL(url);
                                const blankUrl = `${targetUrl.protocol}//${targetUrl.host}`;
                                console.log(`🔧 先导航到空白页面建立域名上下文: ${blankUrl}`);
                                await page.goto(blankUrl, { waitUntil: 'domcontentloaded', timeout: 10000 });
                            } catch (blankNavError) {
                                console.warn(`⚠️ 导航到空白页面失败，直接注入Cookie: ${blankNavError.message}`);
                            }
                            
                            // 逐个注入Cookie
                            let injectedCount = 0;
                            for (const cookie of parsedCookies) {
                                try {
                                    // 构建符合CDP格式的Cookie对象
                                    const cdpCookie = {
                                        name: cookie.name,
                                        value: cookie.value,
                                        domain: cookie.domain,
                                        path: cookie.path || '/',
                                        secure: cookie.secure || false,
                                        httpOnly: cookie.httpOnly || false,
                                        sameSite: cookie.sameSite || 'Lax'
                                    };
                                    
                                    // 设置过期时间
                                    if (cookie.expiry) {
                                        cdpCookie.expires = cookie.expiry;
                                    } else if (cookie.expires) {
                                        cdpCookie.expires = Math.floor(new Date(cookie.expires).getTime() / 1000);
                                    } else {
                                        // 默认1年后过期
                                        cdpCookie.expires = Math.floor(Date.now() / 1000) + 365 * 24 * 60 * 60;
                                    }
                                    
                                    await page.setCookie(cdpCookie);
                                    injectedCount++;
                                    console.log(`✅ Cookie注入成功: ${cookie.name}=${cookie.value.substring(0, 20)}... (域名: ${cookie.domain})`);
                                } catch (cookieError) {
                                    console.warn(`⚠️ Cookie注入失败 ${cookie.name}:`, cookieError.message);
                                }
                            }
                            
                            console.log(`🍪 Cookie动态注入完成: ${injectedCount}/${parsedCookies.length} 成功`);
                            
                            // 验证Cookie是否成功注入
                            try {
                                const currentCookies = await page.cookies();
                                console.log(`🔍 注入后验证: 当前页面共有 ${currentCookies.length} 个Cookie`);
                                if (currentCookies.length > 0) {
                                    console.log(`✅ Cookie验证成功！前3个: ${currentCookies.slice(0, 3).map(c => c.name).join(', ')}`);
                                }
                            } catch (verifyError) {
                                console.warn(`⚠️ Cookie验证失败: ${verifyError.message}`);
                            }
                        }
                        
                        // 🔑 首先注入测试凭证到localStorage（在页面导航之前）
                        console.log('🔑 预先注入测试凭证到localStorage...');
                        try {
                            // 先导航到目标域名的空白页面以建立localStorage上下文
                            const targetUrl = new URL(url);
                            const blankUrl = `${targetUrl.protocol}//${targetUrl.host}`;
                            console.log(`🔧 先导航到空白页面建立localStorage上下文: ${blankUrl}`);
                            await page.goto(blankUrl, { waitUntil: 'domcontentloaded', timeout: 10000 });
                            
                            // 注入配置卡片凭证到localStorage
                            const credentialsResult = await page.evaluate((profile, targetUrl) => {
                                // 从配置卡片生成凭证数据
                                const targetDomain = new URL(targetUrl).hostname;
                                const testCredentials = {};
                                
                                // 如果配置卡片有账号密码，则为目标域名创建凭证
                                if (profile && profile.account && profile.accountPassword) {
                                    testCredentials[targetDomain] = [
                                        {
                                            id: `${profile.id}_${targetDomain}_001`,
                                            username: profile.account,
                                            password: profile.accountPassword,
                                            domain: targetDomain,
                                            url: targetUrl,
                                            createdAt: Date.now(),
                                            lastUsed: Date.now(),
                                            usageCount: 0,
                                            isActive: true
                                        }
                                    ];
                                    
                                    console.log(`✅ 为域名 ${targetDomain} 创建配置卡片凭证: ${profile.account}`);
                                } else {
                                    console.log('⚠️ 配置卡片中没有账号密码信息，跳过凭证创建');
                                    return { success: false, error: '配置卡片中没有账号密码信息' };
                                }
                                
                                // 将凭证保存到localStorage
                                try {
                                    localStorage.setItem('autoLoginCredentials', JSON.stringify(testCredentials));
                                    console.log('✅ 配置卡片凭证已注入到localStorage');
                                    console.log('📊 注入的凭证数据:', testCredentials);
                                    
                                    // 验证localStorage是否成功保存
                                    const saved = localStorage.getItem('autoLoginCredentials');
                                    const parsed = JSON.parse(saved);
                                    return { 
                                        success: true, 
                                        credentialsCount: Object.keys(testCredentials).length,
                                        savedData: parsed
                                    };
                                } catch (error) {
                                    console.error('❌ localStorage注入失败:', error.message);
                                    return { success: false, error: error.message };
                                }
                            }, profile, url);
                            
                            console.log('✅ 本地存储凭证注入结果:', credentialsResult);
                        } catch (localStorageError) {
                            console.error('❌ localStorage注入失败:', localStorageError.message);
                        }
                        
                        // 导航到目标网站
                        console.log(`🌐 导航到目标网站: ${url}`);
                        try {
                            await page.goto(url, { 
                                waitUntil: 'networkidle2',
                                timeout: 30000 
                            });
                            console.log(`✅ 已打开网站: ${url}`);
                            
                            // 🔐 自动填充密码（如果是登录页面）
                            try {
                                console.log(`🔐 尝试自动填充登录表单...`);
                                await autoFillPasswordOnPage(page, finalProfile, profileId);
                            } catch (autoFillError) {
                                console.warn(`⚠️ 自动填充失败: ${autoFillError.message}`);
                            }
                            
                            // 🔐 使用Chrome DevTools Protocol将配置卡片账号密码保存到浏览器原生密码管理器
                            try {
                                console.log('🔐 使用CDP将配置卡片账号密码保存到浏览器原生密码管理器...');
                                
                                // 从profile配置中获取账号密码信息（使用正确的字段名）
                                if (profile && profile.account && profile.accountPassword) {
                                    const currentDomain = new URL(url).hostname;
                                    console.log(`🔑 准备保存账号密码到域名: ${currentDomain}`);
                                    console.log(`🔑 账号: ${profile.account}`);
                                    console.log(`🔒 密码: ${'*'.repeat(profile.accountPassword.length)}`);
                                    
                                    // 等待页面完全加载
                                    await new Promise(resolve => setTimeout(resolve, 3000));
                                    try {
                                        await page.waitForNetworkIdle({ timeout: 5000 });
                                    } catch (e) {
                                        console.log('⚠️ 网络空闲等待超时，继续执行...');
                                    }
                                    await new Promise(resolve => setTimeout(resolve, 2000));
                                    
                                    // 获取CDP客户端
                                    const client = await page.target().createCDPSession();
                                    
                                    try {
                                        // 启用Runtime域
                                        await client.send('Runtime.enable');
                                        
                                        // 使用CDP直接将凭据保存到浏览器密码管理器
                                        const saveCredentialScript = `
                                            (async function() {
                                                try {
                                                    // 创建一个可见的表单来触发浏览器密码保存
                                                    const form = document.createElement('form');
                                                    form.method = 'post';
                                                    form.action = '${url}';
                                                    form.style.position = 'absolute';
                                                    form.style.top = '-9999px';
                                                    form.style.left = '-9999px';
                                                    form.style.opacity = '0';
                                                    form.style.pointerEvents = 'none';
                                                    
                                                    const usernameInput = document.createElement('input');
                                                    usernameInput.type = 'email';
                                                    usernameInput.name = 'username';
                                                    usernameInput.id = 'temp-username-' + Date.now();
                                                    usernameInput.value = '${profile.accountName || profile.account}';
                                                    usernameInput.autocomplete = 'username';
                                                    usernameInput.required = true;
                                                    
                                                    const passwordInput = document.createElement('input');
                                                    passwordInput.type = 'password';
                                                    passwordInput.name = 'password';
                                                    passwordInput.id = 'temp-password-' + Date.now();
                                                    passwordInput.value = '${profile.accountPassword}';
                                                    passwordInput.autocomplete = 'current-password';
                                                    passwordInput.required = true;
                                                    
                                                    const submitButton = document.createElement('button');
                                                    submitButton.type = 'submit';
                                                    submitButton.textContent = 'Login';
                                                    submitButton.id = 'temp-submit-' + Date.now();
                                                    
                                                    form.appendChild(usernameInput);
                                                    form.appendChild(passwordInput);
                                                    form.appendChild(submitButton);
                                                    document.body.appendChild(form);
                                                    
                                                    // 模拟真实的用户输入过程
                                                    usernameInput.focus();
                                                    
                                                    // 模拟逐字符输入用户名
                                                    usernameInput.value = '';
                                                    const username = '${profile.accountName || profile.account}';
                                                    for (let i = 0; i < username.length; i++) {
                                                        usernameInput.value += username[i];
                                                        usernameInput.dispatchEvent(new Event('input', { bubbles: true }));
                                                        await new Promise(resolve => setTimeout(resolve, 50));
                                                    }
                                                    usernameInput.dispatchEvent(new Event('change', { bubbles: true }));
                                                    usernameInput.blur();
                                                    
                                                    await new Promise(resolve => setTimeout(resolve, 300));
                                                    
                                                    // 模拟逐字符输入密码
                                                    passwordInput.focus();
                                                    passwordInput.value = '';
                                                    const password = '${profile.accountPassword}';
                                                    for (let i = 0; i < password.length; i++) {
                                                        passwordInput.value += password[i];
                                                        passwordInput.dispatchEvent(new Event('input', { bubbles: true }));
                                                        await new Promise(resolve => setTimeout(resolve, 50));
                                                    }
                                                    passwordInput.dispatchEvent(new Event('change', { bubbles: true }));
                                                    passwordInput.blur();
                                                    
                                                    await new Promise(resolve => setTimeout(resolve, 500));
                                                    
                                                    // 设置表单提交拦截器
                                                    let submitPrevented = false;
                                                    const preventSubmit = (e) => {
                                                        e.preventDefault();
                                                        e.stopPropagation();
                                                        submitPrevented = true;
                                                        console.log('🔐 表单提交被拦截，触发密码保存检测');
                                                        return false;
                                                    };
                                                    
                                                    form.addEventListener('submit', preventSubmit, true);
                                                    
                                                    // 模拟更真实的用户交互序列
                                                    submitButton.focus();
                                                    
                                                    // 模拟鼠标悬停
                                                    submitButton.dispatchEvent(new MouseEvent('mouseover', { bubbles: true }));
                                                    await new Promise(resolve => setTimeout(resolve, 100));
                                                    
                                                    // 模拟鼠标按下和释放
                                                    submitButton.dispatchEvent(new MouseEvent('mousedown', { bubbles: true }));
                                                    await new Promise(resolve => setTimeout(resolve, 50));
                                                    submitButton.dispatchEvent(new MouseEvent('mouseup', { bubbles: true }));
                                                    
                                                    // 触发点击事件
                                                    submitButton.dispatchEvent(new MouseEvent('click', { bubbles: true }));
                                                    
                                                    // 等待一小段时间，然后手动触发提交事件
                                                    await new Promise(resolve => setTimeout(resolve, 200));
                                                    
                                                    if (!submitPrevented) {
                                                        // 创建更真实的提交事件
                                                        const submitEvent = new Event('submit', { 
                                                            bubbles: true, 
                                                            cancelable: true 
                                                        });
                                                        form.dispatchEvent(submitEvent);
                                                        
                                                        // 如果还是没有触发，尝试直接调用表单的submit方法
                                                        if (!submitPrevented) {
                                                            try {
                                                                form.submit();
                                                            } catch (e) {
                                                                // submit被拦截是正常的
                                                                console.log('🔐 表单提交被正确拦截');
                                                            }
                                                        }
                                                    }
                                                    
                                                    // 等待密码管理器处理
                                                    await new Promise(resolve => setTimeout(resolve, 2000));
                                                    
                                                    // 清理临时表单
                                                    setTimeout(() => {
                                                        if (form.parentNode) {
                                                            form.parentNode.removeChild(form);
                                                        }
                                                    }, 3000);
                                                    
                                                    return { 
                                                        success: true, 
                                                        message: '凭据已注入到浏览器密码管理器检测系统',
                                                        username: username,
                                                        passwordLength: password.length
                                                    };
                                                    
                                                } catch (error) {
                                                    return { 
                                                        success: false, 
                                                        error: error.message 
                                                    };
                                                }
                                            })();
                                        `;
                                        
                                        // 执行脚本
                                        const result = await client.send('Runtime.evaluate', {
                                            expression: saveCredentialScript,
                                            awaitPromise: true,
                                            returnByValue: true
                                        });
                                        
                                        if (result.result && result.result.value) {
                                            const scriptResult = result.result.value;
                                            console.log('🔐 CDP凭据保存结果:', JSON.stringify(scriptResult, null, 2));
                                            
                                            if (scriptResult.success) {
                                                console.log('✅ 凭据已成功注入到浏览器密码管理器检测系统');
                                                console.log(`🔑 保存的账号: ${profile.account}`);
                                                console.log(`🔒 保存的密码长度: ${profile.accountPassword.length} 字符`);
                                                
                                                // 额外尝试：查找页面上的实际登录表单并填充
                                                const fillResult = await page.evaluate(async (account, accountPassword) => {
                                                    const usernameSelectors = [
                                                        'input[type="email"]',
                                                        'input[type="text"][name*="email"]',
                                                        'input[type="text"][name*="username"]',
                                                        'input[name="email"]',
                                                        'input[name="username"]',
                                                        'input[id*="email"]',
                                                        'input[id*="username"]'
                                                    ];
                                                    
                                                    const passwordSelectors = [
                                                        'input[type="password"]',
                                                        'input[name="password"]',
                                                        'input[id*="password"]'
                                                    ];
                                                    
                                                    let usernameField = null;
                                                    let passwordField = null;
                                                    
                                                    // 查找真实的登录表单
                                                    for (const selector of usernameSelectors) {
                                                        const field = document.querySelector(selector);
                                                        if (field && field.offsetParent !== null) {
                                                            usernameField = field;
                                                            break;
                                                        }
                                                    }
                                                    
                                                    for (const selector of passwordSelectors) {
                                                        const field = document.querySelector(selector);
                                                        if (field && field.offsetParent !== null) {
                                                            passwordField = field;
                                                            break;
                                                        }
                                                    }
                                                    
                                                    if (usernameField && passwordField) {
                                                        // 填充真实表单以增强密码管理器检测
                                                        usernameField.value = account;
                                                        usernameField.dispatchEvent(new Event('input', { bubbles: true }));
                                                        
                                                        passwordField.value = accountPassword;
                                                        passwordField.dispatchEvent(new Event('input', { bubbles: true }));
                                                        
                                                        return { success: true, filled: true };
                                                    }
                                                    
                                                    return { success: true, filled: false };
                                                }, profile.account, profile.accountPassword);
                                                
                                                if (fillResult.filled) {
                                                    console.log('✅ 真实登录表单也已填充，增强密码管理器检测');
                                                }
                                                
                                            } else {
                                                console.log('⚠️ CDP凭据保存失败:', scriptResult.error);
                                            }
                                        }
                                        
                                    } catch (cdpError) {
                                        console.error('❌ CDP操作失败:', cdpError.message);
                                        
                                        // 回退到传统方法
                                        console.log('🔄 回退到传统表单填充方法...');
                                        const fallbackResult = await page.evaluate(async (account, accountPassword) => {
                                            // 创建临时表单触发密码保存
                                            const form = document.createElement('form');
                                            form.style.position = 'absolute';
                                            form.style.left = '-9999px';
                                            form.method = 'post';
                                            
                                            const usernameInput = document.createElement('input');
                                            usernameInput.type = 'email';
                                            usernameInput.name = 'username';
                                            usernameInput.value = account;
                                            usernameInput.autocomplete = 'username';
                                            
                                            const passwordInput = document.createElement('input');
                                            passwordInput.type = 'password';
                                            passwordInput.name = 'password';
                                            passwordInput.value = accountPassword;
                                            passwordInput.autocomplete = 'current-password';
                                            
                                            form.appendChild(usernameInput);
                                            form.appendChild(passwordInput);
                                            document.body.appendChild(form);
                                            
                                            // 模拟用户交互
                                            usernameInput.focus();
                                            usernameInput.dispatchEvent(new Event('input', { bubbles: true }));
                                            passwordInput.focus();
                                            passwordInput.dispatchEvent(new Event('input', { bubbles: true }));
                                            
                                            // 触发提交事件
                                            form.dispatchEvent(new Event('submit', { bubbles: true }));
                                            
                                            return { success: true, method: 'fallback' };
                                        }, profile.account, profile.accountPassword);
                                        
                                        console.log('✅ 回退方法执行完成:', fallbackResult);
                                    } finally {
                                        // 清理CDP会话
                                        await client.detach();
                                    }
                                    
                                } else {
                                    console.log('⚠️ 配置卡片中没有账号密码信息');
                                }
                                
                            } catch (credentialError) {
                                console.error('❌ 保存配置卡片账号密码到浏览器密码管理器失败:', credentialError.message);
                            }
                            
                        } catch (navigationError) {
                            console.warn(`⚠️ 打开网站失败 ${url}: ${navigationError.message}`);
                            // Cookie注入成功，网站访问失败不影响浏览器启动
                            continue;
                        }
                        
                        // 🍪 最终验证Cookie状态
                        if (parsedCookies && parsedCookies.length > 0) {
                            try {
                                const finalCookies = await page.cookies();
                                console.log(`🔍 最终Cookie验证: 当前页面共有 ${finalCookies.length} 个Cookie`);
                                if (finalCookies.length > 0) {
                                    console.log(`✅ Cookie最终验证成功！所有Cookie: ${finalCookies.map(c => c.name).join(', ')}`);
                                    
                                    // 检查关键Cookie
                                    const importantCookies = finalCookies.filter(c => 
                                        c.name.includes('c_user') || 
                                        c.name.includes('xs') || 
                                        c.name.includes('datr') || 
                                        c.name.includes('sb')
                                    );
                                    if (importantCookies.length > 0) {
                                        console.log(`🎯 检测到关键Cookie: ${importantCookies.map(c => c.name).join(', ')}`);
                                    }
                                } else {
                                    console.warn(`⚠️ 警告：页面加载后未检测到任何Cookie`);
                                }
                            } catch (finalVerifyError) {
                                console.warn(`⚠️ 最终Cookie验证失败: ${finalVerifyError.message}`);
                            }
                        }
                    } catch (pageError) {
                        console.warn(`⚠️ 打开网站失败 ${url}:`, pageError.message);
                    }
                }
            } else {
                console.log(`ℹ️ 没有启动网站需要打开`);
            }

            return {
                profileId,
                debugPort,
                startTime: browserInstance.createdAt,
                userAgent: config.userAgent,
                proxyConfig: proxy,
                status: 'launched'
            };
        });

        const launchTime = Date.now() - startTime;
        totalLaunchTime += launchTime;
        successCount++;

        console.log(`✅ Puppeteer浏览器启动成功: ${profileId} (${launchTime}ms)`);

        res.json({
            success: true,
            message: 'Puppeteer浏览器启动成功',
            data: result,
            launchTime,
            timestamp: new Date().toISOString()
        });

    } catch (error) {
        const launchTime = Date.now() - startTime;
        errorCount++;
        
        console.error(`❌ 启动Puppeteer浏览器失败: ${profileId}`, error);
        
        res.status(500).json({
            success: false,
            message: 'Puppeteer浏览器启动失败',
            error: error.message,
            profileId,
            launchTime,
            timestamp: new Date().toISOString()
        });
    }
});

// 📋 配置文件管理API
app.get('/api/profiles', (req, res) => {
    try {
        // 返回模拟的配置文件数据，实际应用中应该从数据库读取
        const profiles = [
            {
                id: '1001',
                name: '测试配置1',
                account: 'testuser1',
                password: '****',
                notes: '测试配置',
                proxy: {
                    host: '154.84.31.69',
                    port: '2333',
                    type: 'http'
                },
                status: 'stopped'
            },
            {
                id: '1002',
                name: '测试配置2',
                account: 'testuser2',
                password: '****',
                notes: '测试配置',
                proxy: {
                    host: '192.168.1.100',
                    port: '8080',
                    type: 'http'
                },
                status: 'stopped'
            }
        ];
        
        res.json({
            success: true,
            data: profiles,
            total: profiles.length
        });
    } catch (error) {
        console.error('❌ 获取配置文件失败:', error);
        res.status(500).json({
            success: false,
            error: '获取配置文件失败',
            message: error.message
        });
    }
});

// 📋 获取单个配置文件
app.get('/api/profiles/:id', (req, res) => {
    try {
        const { id } = req.params;
        
        // 模拟数据，实际应用中应该从数据库读取
        const profile = {
            id: id,
            name: `测试配置${id}`,
            account: `testuser${id}`,
            password: '****',
            notes: '测试配置',
            proxy: {
                host: '154.84.31.69',
                port: '2333',
                type: 'http'
            },
            status: 'stopped'
        };
        
        res.json({
            success: true,
            data: profile
        });
    } catch (error) {
        console.error('❌ 获取配置文件失败:', error);
        res.status(500).json({
            success: false,
            error: '获取配置文件失败',
            message: error.message
        });
    }
});

// 📋 创建配置文件
app.post('/api/profiles', (req, res) => {
    try {
        const profileData = req.body;
        console.log('🔄 创建配置文件:', profileData);
        
        // 模拟创建成功
        const newProfile = {
            id: Date.now().toString(),
            ...profileData,
            status: 'stopped',
            createdAt: new Date().toISOString()
        };
        
        res.json({
            success: true,
            data: newProfile,
            message: '配置文件创建成功'
        });
    } catch (error) {
        console.error('❌ 创建配置文件失败:', error);
        res.status(500).json({
            success: false,
            error: '创建配置文件失败',
            message: error.message
        });
    }
});

// 📋 会话管理API
app.get('/api/sessions', (req, res) => {
    try {
        const sessions = Array.from(activeBrowsers.entries()).map(([profileId, browserData]) => ({
            profileId,
            port: browserData.config?.debugPort,
            status: browserData.browser?.isConnected() ? 'active' : 'inactive',
            startTime: browserData.startTime,
            lastUsed: browserData.lastUsed
        }));
        
        res.json({
            success: true,
            data: sessions,
            total: sessions.length
        });
    } catch (error) {
        console.error('❌ 获取会话列表失败:', error);
        res.status(500).json({
            success: false,
            error: '获取会话列表失败',
            message: error.message
        });
    }
});

// 🗑️ 清理端点
app.post('/api/cleanup', async (req, res) => {
    try {
        console.log('🧹 开始清理资源...');
        
        // 清理空闲浏览器实例
        await browserManager.cleanupIdleInstances();
        
        // 强制垃圾回收
        if (global.gc) {
            global.gc();
        }
        
        console.log('✅ 资源清理完成');
        
        res.json({
            success: true,
            message: '资源清理完成',
            timestamp: new Date().toISOString(),
            stats: browserManager.getStats()
        });
        
    } catch (error) {
        console.error('❌ 清理失败:', error);
        
        res.status(500).json({
            success: false,
            message: '清理失败',
            error: error.message
        });
    }
});

// 🔧 DevTools代理API - 用于线上域名访问本地DevTools
app.get('/api/devtools/:debugPort/json/list', async (req, res) => {
    try {
        const { debugPort } = req.params;
        const http = require('http');
        
        console.log(`🔧 DevTools代理请求: localhost:${debugPort}/json/list`);
        
        const options = {
            hostname: 'localhost',
            port: debugPort,
            path: `/json/list?t=${Date.now()}`,
            method: 'GET'
        };
        
        const request = http.request(options, (response) => {
            let data = '';
            response.on('data', (chunk) => {
                data += chunk;
            });
            response.on('end', () => {
                try {
                    const jsonData = JSON.parse(data);
                    res.json(jsonData);
                } catch (parseError) {
                    console.error('❌ JSON解析失败:', parseError.message);
                    res.status(500).json({
                        success: false,
                        error: 'JSON解析失败',
                        message: parseError.message
                    });
                }
            });
        });
        
        request.on('error', (error) => {
            console.error(`❌ DevTools代理失败:`, error.message);
            res.status(500).json({
                success: false,
                error: 'DevTools代理失败',
                message: error.message
            });
        });
        
        request.end();
    } catch (error) {
        console.error(`❌ DevTools代理失败:`, error.message);
        res.status(500).json({
            success: false,
            error: 'DevTools代理失败',
            message: error.message
        });
    }
});

app.get('/api/devtools/:debugPort/json/version', async (req, res) => {
    try {
        const { debugPort } = req.params;
        const http = require('http');
        
        console.log(`🔧 DevTools代理请求: localhost:${debugPort}/json/version`);
        
        const options = {
            hostname: 'localhost',
            port: debugPort,
            path: '/json/version',
            method: 'GET'
        };
        
        const request = http.request(options, (response) => {
            let data = '';
            response.on('data', (chunk) => {
                data += chunk;
            });
            response.on('end', () => {
                try {
                    const jsonData = JSON.parse(data);
                    res.json(jsonData);
                } catch (parseError) {
                    console.error('❌ JSON解析失败:', parseError.message);
                    res.status(500).json({
                        success: false,
                        error: 'JSON解析失败',
                        message: parseError.message
                    });
                }
            });
        });
        
        request.on('error', (error) => {
            console.error(`❌ DevTools代理失败:`, error.message);
            res.status(500).json({
                success: false,
                error: 'DevTools代理失败',
                message: error.message
            });
        });
        
        request.end();
    } catch (error) {
        console.error(`❌ DevTools代理失败:`, error.message);
        res.status(500).json({
            success: false,
            error: 'DevTools代理失败',
            message: error.message
        });
    }
});

// 🔧 DevTools WebSocket代理 - 用于Runtime.evaluate等操作
app.post('/api/devtools/:debugPort/runtime/evaluate', async (req, res) => {
    try {
        const { debugPort } = req.params;
        const { expression, awaitPromise = false, returnByValue = true } = req.body;
        const WebSocket = require('ws');
        
        console.log(`🔧 DevTools Runtime.evaluate代理: localhost:${debugPort}`);
        
        // 首先获取可用的targets
        const http = require('http');
        const targets = await new Promise((resolve, reject) => {
            const options = {
                hostname: 'localhost',
                port: debugPort,
                path: `/json/list?t=${Date.now()}`,
                method: 'GET'
            };
            
            const request = http.request(options, (response) => {
                let data = '';
                response.on('data', (chunk) => {
                    data += chunk;
                });
                response.on('end', () => {
                    try {
                        const jsonData = JSON.parse(data);
                        resolve(jsonData);
                    } catch (parseError) {
                        reject(parseError);
                    }
                });
            });
            
            request.on('error', (error) => {
                reject(error);
            });
            
            request.end();
        });
        
        if (!Array.isArray(targets) || targets.length === 0) {
            throw new Error('未找到可用的调试页面');
        }
        
        const target = targets.find(item => item.type === 'page' && item.url && !item.url.startsWith('devtools://') && !item.url.startsWith('chrome-extension://')) ||
                       targets.find(item => item.webSocketDebuggerUrl);
        
        if (!target || !target.webSocketDebuggerUrl) {
            throw new Error('调试目标缺少 WebSocket 地址');
        }
        
        // 通过WebSocket执行Runtime.evaluate
        const result = await new Promise((resolve, reject) => {
            let settled = false;
            let evalMessageId = null;
            let messageId = 1;
            const socket = new WebSocket(target.webSocketDebuggerUrl);
            
            const cleanup = () => {
                if (socket.readyState === WebSocket.OPEN || socket.readyState === WebSocket.CONNECTING) {
                    socket.close();
                }
            };
            
            const timeout = setTimeout(() => {
                if (!settled) {
                    settled = true;
                    cleanup();
                    reject(new Error('Runtime.evaluate 超时'));
                }
            }, 8000);
            
            socket.onopen = () => {
                socket.send(JSON.stringify({ id: messageId++, method: 'Runtime.enable' }));
                evalMessageId = messageId++;
                socket.send(JSON.stringify({
                    id: evalMessageId,
                    method: 'Runtime.evaluate',
                    params: {
                        expression: expression,
                        awaitPromise: awaitPromise,
                        returnByValue: returnByValue
                    }
                }));
            };
            
            socket.onmessage = event => {
                if (settled) return;
                
                try {
                    const data = JSON.parse(event.data);
                    if (data.id === evalMessageId) {
                        settled = true;
                        clearTimeout(timeout);
                        cleanup();
                        
                        if (data.error) {
                            reject(new Error(data.error.message || 'Runtime.evaluate 执行失败'));
                            return;
                        }
                        
                        resolve(data.result);
                    }
                } catch (parseError) {
                    console.warn('⚠️ WebSocket消息解析失败:', parseError.message);
                }
            };
            
            socket.onerror = error => {
                if (!settled) {
                    settled = true;
                    clearTimeout(timeout);
                    cleanup();
                    reject(new Error(`WebSocket连接失败: ${error.message}`));
                }
            };
            
            socket.onclose = () => {
                if (!settled) {
                    settled = true;
                    clearTimeout(timeout);
                    reject(new Error('WebSocket连接意外关闭'));
                }
            };
        });
        
        res.json({
            success: true,
            result: result
        });
        
    } catch (error) {
        console.error(`❌ DevTools Runtime.evaluate代理失败:`, error.message);
        res.status(500).json({
            success: false,
            error: 'DevTools Runtime.evaluate代理失败',
            message: error.message
        });
    }
});

// ⚠️ 错误处理中间件
app.use((err, req, res, next) => {
    console.error('🔥 服务器错误:', err);
    
    res.status(500).json({
        success: false,
        message: '服务器内部错误',
        error: err.message,
        timestamp: new Date().toISOString()
    });
});

// 🚀 服务器启动逻辑已移至文件末尾

// 🕐 定期清理任务 - 已禁用自动清理功能
// setInterval(async () => {
//     try {
//         await browserManager.cleanupIdleInstances();
//         
//         // 内存使用检查
//         const memUsage = process.memoryUsage();
//         const memMB = Math.round(memUsage.heapUsed / 1024 / 1024);
//         
//         if (memMB > 500) { // 超过500MB时警告
//             console.warn(`⚠️ 内存使用较高: ${memMB}MB`);
//             
//             // 强制垃圾回收
//             if (global.gc) {
//                 global.gc();
//                 console.log('🧹 已执行垃圾回收');
//             }
//         }
//         
//     } catch (error) {
//         console.error('❌ 定期清理任务失败:', error);
//     }
// }, MEMORY_CLEANUP_INTERVAL);

console.log('ℹ️ 浏览器自动清理功能已禁用，浏览器实例将保持运行直到手动关闭');

// 🛡️ 优雅关闭处理
process.on('SIGINT', async () => {
    console.log('\n🔄 正在优雅关闭 Puppeteer 服务...');
    
    try {
        // 关闭所有浏览器实例
        console.log('🗑️ 正在关闭所有浏览器实例...');
        await browserManager.cleanup();
        
        // 清理活跃浏览器
        for (const [profileId, browserData] of activeBrowsers) {
            try {
                if (browserData.browser && browserData.browser.isConnected()) {
                    await browserData.browser.close();
                }
            } catch (error) {
                console.warn(`⚠️ 关闭浏览器失败 ${profileId}:`, error.message);
            }
        }
        
        activeBrowsers.clear();
        
        console.log('✅ 资源清理完成');
        console.log('👋 Puppeteer 服务已关闭');
        
    } catch (error) {
        console.error('❌ 关闭过程中出现错误:', error);
    }
    
    process.exit(0);
});

// 🍪 Cookie预注入函数 - 在浏览器启动前将Cookie写入用户数据目录
async function preinjectCookies(userDataDir, cookies) {
    const fs = require('fs');
    const path = require('path');
    const sqlite3 = require('sqlite3').verbose();
    
    try {
        // 确保用户数据目录存在
        if (!fs.existsSync(userDataDir)) {
            fs.mkdirSync(userDataDir, { recursive: true });
            console.log(`📁 创建用户数据目录: ${userDataDir}`);
        }
        
        // 创建Default目录
        const defaultDir = path.join(userDataDir, 'Default');
        if (!fs.existsSync(defaultDir)) {
            fs.mkdirSync(defaultDir, { recursive: true });
            console.log(`📁 创建Default目录: ${defaultDir}`);
        }
        
        // Cookie数据库路径
        const cookieDbPath = path.join(defaultDir, 'Cookies');
        console.log(`🗄️ Cookie数据库路径: ${cookieDbPath}`);
        
        // 创建或打开Cookie数据库
        const db = new sqlite3.Database(cookieDbPath);
        
        // 创建cookies表（如果不存在）
        await new Promise((resolve, reject) => {
            db.run(`CREATE TABLE IF NOT EXISTS cookies (
                creation_utc INTEGER NOT NULL,
                host_key TEXT NOT NULL,
                top_frame_site_key TEXT NOT NULL,
                name TEXT NOT NULL,
                value TEXT NOT NULL,
                encrypted_value BLOB DEFAULT '',
                path TEXT NOT NULL,
                expires_utc INTEGER NOT NULL,
                is_secure INTEGER NOT NULL,
                is_httponly INTEGER NOT NULL,
                last_access_utc INTEGER NOT NULL,
                has_expires INTEGER NOT NULL,
                is_persistent INTEGER NOT NULL,
                priority INTEGER NOT NULL DEFAULT 1,
                samesite INTEGER NOT NULL DEFAULT -1,
                source_scheme INTEGER NOT NULL DEFAULT 0,
                source_port INTEGER NOT NULL DEFAULT -1,
                is_same_party INTEGER NOT NULL DEFAULT 0
            )`, (err) => {
                if (err) reject(err);
                else resolve();
            });
        });
        
        // 插入Cookie数据
        const currentTime = Date.now() * 1000; // Chrome时间戳（微秒）
        const farFuture = currentTime + (10 * 365 * 24 * 60 * 60 * 1000 * 1000); // 10年后
        
        for (const cookie of cookies) {
            try {
                // 计算过期时间
                let expiresUtc = farFuture;
                if (cookie.expiry) {
                    expiresUtc = cookie.expiry * 1000 * 1000; // 转换为Chrome时间戳
                } else if (cookie.expires) {
                    expiresUtc = new Date(cookie.expires).getTime() * 1000;
                }
                
                // 确定域名和路径
                const domain = cookie.domain || cookie.host_key || '.localhost';
                const path = cookie.path || '/';
                
                // 插入Cookie
                await new Promise((resolve, reject) => {
                    db.run(`INSERT OR REPLACE INTO cookies (
                        creation_utc, host_key, top_frame_site_key, name, value,
                        path, expires_utc, is_secure, is_httponly, last_access_utc,
                        has_expires, is_persistent, priority, samesite
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`, [
                        currentTime,
                        domain,
                        domain,
                        cookie.name,
                        cookie.value,
                        path,
                        expiresUtc,
                        cookie.secure ? 1 : 0,
                        cookie.httpOnly ? 1 : 0,
                        currentTime,
                        1, // has_expires
                        1, // is_persistent
                        1, // priority
                        cookie.sameSite === 'Strict' ? 1 : (cookie.sameSite === 'Lax' ? 0 : -1)
                    ], (err) => {
                        if (err) reject(err);
                        else resolve();
                    });
                });
                
                console.log(`✅ Cookie已注入: ${cookie.name} = ${cookie.value.substring(0, 20)}...`);
            } catch (cookieError) {
                console.warn(`⚠️ 注入Cookie失败 ${cookie.name}:`, cookieError.message);
            }
        }
        
        // 关闭数据库
        db.close();
        console.log(`🍪 Cookie预注入完成，共注入 ${cookies.length} 个Cookie`);
        
    } catch (error) {
        console.error(`❌ Cookie预注入失败:`, error.message);
        throw error;
    }
}

// 🍪 Cookie预注入到本地Chrome数据库
async function preinjectCookiesToDatabase(userDataDir, cookies) {
    if (!cookies || cookies.length === 0) {
        console.log('ℹ️ 没有Cookie需要预注入');
        return true;
    }

    const cookieDbPath = path.join(userDataDir, 'Default', 'Cookies');
    const networkDir = path.join(userDataDir, 'Default', 'Network');
    
    try {
        // 确保目录存在
        await fs.mkdir(path.join(userDataDir, 'Default'), { recursive: true });
        await fs.mkdir(networkDir, { recursive: true });
        
        console.log(`🍪 开始预注入 ${cookies.length} 个Cookie到: ${cookieDbPath}`);
        
        // 检查Cookie数据库是否已存在
        let dbExists = false;
        try {
            const stats = await fs.stat(cookieDbPath);
            dbExists = stats.isFile();
            console.log('✅ Cookie数据库已存在，将追加Cookie');
        } catch (error) {
            dbExists = false;
            console.log('📝 Cookie数据库不存在，将创建新数据库');
        }
        
        return new Promise((resolve, reject) => {
            console.log(`🔧 正在打开SQLite数据库: ${cookieDbPath}`);
            const db = new sqlite3.Database(cookieDbPath, (err) => {
                if (err) {
                    console.error('❌ 打开Cookie数据库失败:', err);
                    reject(err);
                    return;
                }
                console.log('✅ SQLite数据库连接成功');
                
                // 如果是新数据库，需要创建表结构
                if (!dbExists) {
                    console.log('🔧 创建Chrome Cookie表结构...');
                    db.run(`CREATE TABLE cookies (
                        creation_utc INTEGER NOT NULL,
                        host_key TEXT NOT NULL,
                        top_frame_site_key TEXT NOT NULL,
                        name TEXT NOT NULL,
                        value TEXT NOT NULL,
                        encrypted_value BLOB DEFAULT '',
                        path TEXT NOT NULL,
                        expires_utc INTEGER NOT NULL,
                        is_secure INTEGER NOT NULL DEFAULT 0,
                        is_httponly INTEGER NOT NULL DEFAULT 0,
                        last_access_utc INTEGER NOT NULL,
                        has_expires INTEGER NOT NULL DEFAULT 1,
                        is_persistent INTEGER NOT NULL DEFAULT 1,
                        priority INTEGER NOT NULL DEFAULT 1,
                        samesite INTEGER NOT NULL DEFAULT -1,
                        source_scheme INTEGER NOT NULL DEFAULT 0,
                        source_port INTEGER NOT NULL DEFAULT -1,
                        last_update_utc INTEGER NOT NULL,
                        UNIQUE (host_key, name, path)
                    )`, (err) => {
                        if (err) {
                            console.error('❌ 创建Cookie表失败:', err);
                            reject(err);
                            return;
                        }
                        console.log('✅ Cookie表结构创建成功');
                        insertCookies();
                    });
                } else {
                    insertCookies();
                }
                
                function insertCookies() {
                    const now = Date.now() * 1000; // Chrome uses microseconds
                    let insertedCount = 0;
                    let errorCount = 0;
                    
                    const stmt = db.prepare(`INSERT OR REPLACE INTO cookies (
                        creation_utc, host_key, top_frame_site_key, name, value, encrypted_value,
                        path, expires_utc, is_secure, is_httponly, last_access_utc,
                        has_expires, is_persistent, priority, samesite, source_scheme,
                        source_port, last_update_utc
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`);

                    cookies.forEach(cookie => {
                        // 计算过期时间
                        const expiresUtc = cookie.expiry ? 
                            cookie.expiry * 1000000 : // 转换为微秒
                            (now + 365 * 24 * 60 * 60 * 1000000); // 默认1年后过期
                        
                        // 处理domain为host_key
                        const hostKey = cookie.domain || '.example.com';
                        const topFrameSiteKey = hostKey.startsWith('.') ? 
                            'https://' + hostKey.substring(1) : 
                            'https://' + hostKey;
                        
                        // SameSite值映射
                        let sameSiteValue = -1; // UNSPECIFIED
                        if (cookie.sameSite === 'None') sameSiteValue = 0;
                        else if (cookie.sameSite === 'Lax') sameSiteValue = 1;
                        else if (cookie.sameSite === 'Strict') sameSiteValue = 2;
                        
                        stmt.run([
                            now,                           // creation_utc
                            hostKey,                       // host_key
                            topFrameSiteKey,              // top_frame_site_key
                            cookie.name,                   // name
                            cookie.value,                  // value
                            '',                           // encrypted_value (空字符串)
                            cookie.path || '/',           // path
                            expiresUtc,                   // expires_utc
                            cookie.secure ? 1 : 0,        // is_secure
                            cookie.httpOnly ? 1 : 0,      // is_httponly
                            now,                          // last_access_utc
                            1,                            // has_expires
                            1,                            // is_persistent
                            1,                            // priority (MEDIUM)
                            sameSiteValue,                // samesite
                            2,                            // source_scheme (SECURE)
                            443,                          // source_port
                            now                           // last_update_utc
                        ], function(err) {
                            if (err) {
                                console.error(`❌ 插入Cookie失败 ${cookie.name}:`, err);
                                errorCount++;
                            } else {
                                console.log(`✅ Cookie已插入: ${cookie.name}=${cookie.value.substring(0, 20)}...`);
                                insertedCount++;
                            }
                            
                            // 检查是否所有Cookie都已处理
                            if (insertedCount + errorCount === cookies.length) {
                                stmt.finalize();
                                db.close(async (err) => {
                                    if (err) {
                                        console.error('❌ 关闭数据库失败:', err);
                                        reject(err);
                                    } else {
                                        console.log(`✅ Cookie预注入完成: ${insertedCount}/${cookies.length} 成功`);
                                        
                                        // 验证Cookie文件是否真正创建
                                        try {
                                            const stats = await fs.stat(cookieDbPath);
                                            console.log(`🔍 Cookie数据库文件验证: 大小=${stats.size}字节, 修改时间=${stats.mtime}`);
                                        } catch (verifyError) {
                                            console.error('❌ Cookie数据库文件验证失败:', verifyError);
                                        }
                                        
                                        resolve(true);
                                    }
                                });
                            }
                        });
                    });
                }
            });
        });
        
    } catch (error) {
        console.error('❌ Cookie预注入失败:', error);
        throw error;
    }
}

// 🍪 Cookie字符串解析函数
function parseCookieString(cookieString, startUrls = []) {
    if (!cookieString || typeof cookieString !== 'string') {
        return [];
    }
    
    const cookies = [];
    
    // 🔧 智能域名推导：根据启动网站自动设置cookie域名
    let defaultDomain = '.example.com';
    if (startUrls && startUrls.length > 0) {
        try {
            const firstUrl = new URL(startUrls[0]);
            const hostname = firstUrl.hostname;
            // 如果是主域名，添加点前缀以支持子域名
            defaultDomain = hostname.startsWith('www.') ? 
                '.' + hostname.substring(4) : 
                '.' + hostname;
            console.log(`🔧 根据启动网站自动推导Cookie默认域名: ${defaultDomain} (基于: ${hostname})`);
        } catch (urlError) {
            console.warn('⚠️ 无法解析启动网站URL，使用默认域名:', defaultDomain);
        }
    }
    
    // 🔧 检查是否为HTTPS网站（影响secure和sameSite设置）
    const isHttps = startUrls && startUrls.length > 0 && startUrls[0].startsWith('https://');
    console.log(`🔒 启动网站协议检测: ${isHttps ? 'HTTPS (将设置secure=true)' : 'HTTP (secure=false)'}`);
    
    try {
        // 尝试解析JSON格式的Cookie
        if (cookieString.trim().startsWith('[') || cookieString.trim().startsWith('{')) {
            const jsonCookies = JSON.parse(cookieString);
            if (Array.isArray(jsonCookies)) {
                return jsonCookies;
            } else if (typeof jsonCookies === 'object') {
                // 单个cookie对象
                return [jsonCookies];
            }
        }
        
        // 解析Netscape格式或简单的键值对格式
        const lines = cookieString.split('\n').map(line => line.trim()).filter(line => line);
        
        for (const line of lines) {
            // 跳过注释行
            if (line.startsWith('#') || line.startsWith('//')) {
                continue;
            }
            
            // 检查是否是Netscape格式（包含制表符）
            if (line.includes('\t')) {
                const parts = line.split('\t');
                if (parts.length >= 7) {
                    const cookie = {
                        domain: parts[0],
                        path: parts[2] || '/',
                        secure: parts[3] === 'TRUE',
                        expiry: parseInt(parts[4]) || Math.floor(Date.now() / 1000) + 86400 * 365, // 默认1年
                        name: parts[5],
                        value: parts[6] || '',
                        httpOnly: false,
                        sameSite: 'Lax'
                    };
                    cookies.push(cookie);
                    continue;
                }
            }
            
            // 解析简单的键值对格式 (name=value; name2=value2)
            if (line.includes('=')) {
                const cookiePairs = line.split(';').map(pair => pair.trim());
                
                for (const pair of cookiePairs) {
                    const equalIndex = pair.indexOf('=');
                    if (equalIndex > 0) {
                        const name = pair.substring(0, equalIndex).trim();
                        const value = pair.substring(equalIndex + 1).trim();
                        
                        if (name && value) {
                            const cookie = {
                                domain: defaultDomain,
                                path: '/',
                                secure: isHttps, // 根据启动网站协议自动设置
                                expiry: Math.floor(Date.now() / 1000) + 86400 * 365, // 1年后过期
                                name: name,
                                value: value,
                                httpOnly: false,
                                sameSite: isHttps ? 'None' : 'Lax' // HTTPS使用None，HTTP使用Lax
                            };
                            cookies.push(cookie);
                            console.log(`🍪 解析Cookie: ${name} -> 域名: ${defaultDomain}, secure: ${isHttps}, sameSite: ${cookie.sameSite}`);
                        }
                    }
                }
            }
        }
        
    } catch (parseError) {
        console.error('❌ Cookie解析失败:', parseError.message);
        console.log('🔍 原始Cookie数据:', cookieString.substring(0, 200));
    }
    
    console.log(`🔧 Cookie解析完成: ${cookies.length} 个cookie`);
    return cookies;
}

// 🚀 启动优化后的服务器
let server;

// 检查是否启用 HTTPS
const ENABLE_HTTPS = process.env.ENABLE_HTTPS === 'true' || process.env.NODE_ENV === 'production';
const HTTPS_PORT = process.env.HTTPS_PORT || 9443;

if (ENABLE_HTTPS) {
    try {
        // 读取 SSL 证书
        const privateKey = fsSync.readFileSync(path.join(__dirname, 'puppeteer-key.pem'), 'utf8');
        const certificate = fsSync.readFileSync(path.join(__dirname, 'puppeteer-cert.pem'), 'utf8');
        
        const credentials = { key: privateKey, cert: certificate };
        
        // 创建 HTTPS 服务器
        server = https.createServer(credentials, app);
        server.listen(HTTPS_PORT, '0.0.0.0', () => {
            const actualPort = server.address().port;
            log('INFO', `🚀 Puppeteer HTTPS 服务已启动在端口 ${actualPort}`);
            log('INFO', `📊 健康检查: https://localhost:${actualPort}/health`);
            log('INFO', `🌐 启动浏览器: https://localhost:${actualPort}/api/launch-browser`);
            log('INFO', `📈 性能监控: https://localhost:${actualPort}/api/stats`);
            log('INFO', `🎯 日志级别: ${process.env.LOG_LEVEL || 'INFO'}`);
            log('INFO', `🔒 HTTPS 模式已启用`);
            
            if (actualPort !== HTTPS_PORT) {
                log('WARN', `⚠️ 警告: 服务启动在端口 ${actualPort}，而不是预期的 ${HTTPS_PORT}`);
            }
        });
    } catch (error) {
        log('ERROR', `❌ HTTPS 启动失败: ${error.message}`);
        log('INFO', `🔄 回退到 HTTP 模式...`);
        
        // 回退到 HTTP
        server = app.listen(PORT, '0.0.0.0', () => {
            const actualPort = server.address().port;
            log('INFO', `🚀 Puppeteer HTTP 服务已启动在端口 ${actualPort}`);
            log('INFO', `📊 健康检查: http://localhost:${actualPort}/health`);
            log('INFO', `🌐 启动浏览器: http://localhost:${actualPort}/api/launch-browser`);
            log('INFO', `📈 性能监控: http://localhost:${actualPort}/api/stats`);
            log('INFO', `🎯 日志级别: ${process.env.LOG_LEVEL || 'INFO'}`);
            
            if (actualPort !== PORT) {
                log('WARN', `⚠️ 警告: 服务启动在端口 ${actualPort}，而不是预期的 ${PORT}`);
            }
        });
    }
} else {
    // HTTP 模式
    server = app.listen(PORT, '0.0.0.0', () => {
        const actualPort = server.address().port;
        log('INFO', `🚀 Puppeteer HTTP 服务已启动在端口 ${actualPort}`);
        log('INFO', `📊 健康检查: http://localhost:${actualPort}/health`);
        log('INFO', `🌐 启动浏览器: http://localhost:${actualPort}/api/launch-browser`);
        log('INFO', `📈 性能监控: http://localhost:${actualPort}/api/stats`);
        log('INFO', `🎯 日志级别: ${process.env.LOG_LEVEL || 'INFO'}`);
        
        if (actualPort !== PORT) {
            log('WARN', `⚠️ 警告: 服务启动在端口 ${actualPort}，而不是预期的 ${PORT}`);
        }
    });
}

server.on('error', (err) => {
    const currentPort = ENABLE_HTTPS ? HTTPS_PORT : PORT;
    if (err.code === 'EADDRINUSE') {
        log('ERROR', `❌ 端口 ${currentPort} 已被占用，请检查其他服务`);
        process.exit(1);
    } else {
        log('ERROR', `❌ 服务器启动失败: ${err.message}`);
        process.exit(1);
    }
});
